//
//  SQLiteConnection.m
//  APDA
//
//  Created by Jonathan Diehl on 12.05.08.
//  Copyright 2008 Media Computing Group. All rights reserved.
//

#import "SQLiteConnection.h"
#import "SQLiteStatement.h"

@implementation SQLiteConnection

// Constructor
- (id)initWithDatabase:(NSString *)newDatabaseName;
{
	self = [super init];
	if (self != nil) {
		databaseName = newDatabaseName;
		[self open];
	}
	return self;
}

// Open the database (create new database is necessary)
- (void)open;
{

    // First, test for existence of the databasefile
    NSFileManager *fileManager = [NSFileManager defaultManager];
    NSError *error;
    NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
    NSString *documentsDirectory = [paths objectAtIndex:0];
    NSString *path = [documentsDirectory stringByAppendingPathComponent:databaseName];
	
	// The writable database does not exist, so copy the default to the appropriate location.
    if(![fileManager fileExistsAtPath:path]) {
		NSString *defaultDBPath = [[[NSBundle mainBundle] resourcePath] stringByAppendingPathComponent:databaseName];
		if(![fileManager copyItemAtPath:defaultDBPath toPath:path error:&error]) {
			NSAssert1(0, @"Failed to create writable database file with message '%@'.", [error localizedDescription]);
			return;
		}
	}
	
	// Database file is created and editable, so open the database
    if (sqlite3_open([path UTF8String], &database) != SQLITE_OK) {
        // Even though the open failed, call close to properly clean up resources.
        sqlite3_close(database);
        NSAssert1(0, @"Failed to open database with message '%s'.", sqlite3_errmsg(database));
		return;
	}
}

// Close the database
- (void)close;
{
	sqlite3_close(database);
}

// Perform a query
- (DatabaseStatement *)selectQuery:(NSString *)query;
{
	SQLiteStatement *statement = [[[SQLiteStatement alloc] initWithConnection:self query:query] autorelease];
	return statement;
}

// Perform a query
- (void)updateQuery:(NSString *)query;
{
	SQLiteStatement *statement = [[SQLiteStatement alloc] initWithConnection:self query:query];
	[statement next];
	[statement finalize];
	[statement release];
}

// Prepare a query and return the statement
- (sqlite3_stmt *)prepareQuery:(NSString *)query;
{
	NSLog(@"SQL: %@", query);

	sqlite3_stmt *statement;

	// Preparing a statement compiles the SQL query into a byte-code program in the SQLite library.
	// The third parameter is either the length of the SQL string or -1 to read up to the first null terminator.        
	if(sqlite3_prepare_v2(database, [query UTF8String], -1, &statement, NULL) != SQLITE_OK) {
		NSAssert1(0, @"Query failed with message '%s'.", sqlite3_errmsg(database));
		return nil;
	}
	
	return statement;
}


- (void)dealloc;
{
	[self close];
	[super dealloc];
}

@end
