//
//  ActiveRecord.m
//  APDA
//
//  Created by Jonathan Diehl on 12.05.08.
//  Copyright 2008 Media Computing Group. All rights reserved.
//

#import "ActiveRecord.h"
#import <sqlite3.h>
#import "DatabaseStatement.h"
#import "DatabaseColumn.h"


// private methods
@interface ActiveRecord (PrivateMethods)

+ (int)getNewId;

@end

@implementation ActiveRecord

@synthesize idValue;

// retrieve the connection
+ (DatabaseConnection *)connection;
{
	return [DatabaseConnection connection];
}

// retrieve the table name
+ (NSString *)tableName;
{
	return [self className];
}

// find records by giving an sql query
+ (NSArray *)findBySQL:(NSString *)query;
{
	DatabaseConnection *connection = [self connection];
	DatabaseStatement *statement = [[connection selectQuery:query] retain];
	
	NSMutableArray *records = [NSMutableArray arrayWithCapacity:[statement count]];
	while([statement next])
	{
		ActiveRecord *record = [[[self class] alloc] initWithStatement:statement];
		[records addObject:record];
		[record release];
	}

	// clean up
	[statement release];
	
	return records;
}

// find all records
+ (NSArray *)findAll;
{
	return [self findBySQL:[NSString stringWithFormat:@"select * from `%@`", [self tableName]]];
}

// find by id
+ (ActiveRecord *)findById:(int)value;
{
	NSString *query = [NSString stringWithFormat:@"select * from `%@` where id=%d", [self tableName], value];
	NSArray *items = [self findBySQL:query];
	return [items objectAtIndex:0];
}

// get new id
+ (int)getNewId;
{
	NSString *query = [NSString stringWithFormat:@"select max(id) from `%@`", [self tableName]];
	DatabaseStatement *stmt = [[self connection] selectQuery:query];
	
	// try to read the first row
	if(![stmt next]) {
		return 1;
	}
	
	int lastId = [stmt intValueForIndex:0];
	return lastId + 1;
}

// Constructor
- (id)initWithStatement:(DatabaseStatement *)statement;
{
	self = [super init];
	if (self != nil) {
		[self setIdValue:[statement intValueForIndex:0]];
	}
	return self;
}

// get the table name
- (NSString *)tableName;
{
	return [[self class] tableName];
}

// get the connection
- (DatabaseConnection *)connection;
{
	return [[self class] connection];
}

// update or create the record
- (void)save;
{
	DatabaseConnection *connection = [self connection];
	
	if([self idValue] == 0) {
		// get new id and insert
		[self setIdValue:[[self class] getNewId]];
		[connection updateQuery:[self insertQuery]];
	} else {
		// update
		[connection updateQuery:[self updateQuery]];
	}
}

// delete the record
- (void)destroy;
{
	if([self idValue] != 0) {
		[[self connection] updateQuery:[self deleteQuery]];
	}
}

// query for inserting records
- (NSString *)insertQuery;
{
	return [NSString stringWithFormat:@"insert into `%@` (id) values (%i)", [self tableName], [self idValue]];
}

// query for updating records
- (NSString *)updateQuery;
{
	return [NSString stringWithFormat:@"update `%@` set id=%i", [self tableName], [self idValue]];
}

// query for deleting records
- (NSString *)deleteQuery;
{
	return [NSString stringWithFormat:@"delete from `%@` where id=%i", [self tableName], [self idValue]];
}

// retrieve the identifier for this class's table view cell
- (NSString *)tableViewCellIdentifier;
{
	return [self className];
}

// table view cell representation for an object
- (UITableViewCell *)tableViewCell;
{
	UITableViewCell *cell = [[[UITableViewCell alloc] initWithFrame:CGRectZero 
													reuseIdentifier:[self tableViewCellIdentifier]] autorelease];
	[cell setAccessoryType:UITableViewCellAccessoryDisclosureIndicator];
	return cell;
}

// fill the cell with content
- (void)updateTableViewCell:(UITableViewCell *)cell;
{
	[cell setText:[NSString stringWithFormat:@"%@ #%i", [self className], [self idValue]]];
}

// get the cell for a table view
- (UITableViewCell *)cellForTableView:(UITableView *)tableView;
{
	UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:[self tableViewCellIdentifier]];
	if(cell == NULL) {
		cell = [self tableViewCell];
	}
	[self updateTableViewCell:cell];
	return cell;
}

@end
