//
//  Temp.swift
//  TemperatureConverter
//
//  Created by Nur Al-huda Hamdan on 09/11/15.
//  Copyright © 2015 RWTH Aachen Univesity. All rights reserved.
//

import Foundation

typealias ConversionFunction = (Double) -> Double

infix operator ∘ {associativity right precedence 150}


func ∘ (lhs: ConversionFunction, rhs: ConversionFunction) -> ConversionFunction {
    return {(value : Double) -> Double in
        return lhs(rhs(value))}
}




enum TemperatureUnit : String {
    case celsius = "˚C"
    case fahrenheit = "˚F"
    case kelvin = "˚K"
    
    func conversionFunction(toUnit targetUnit: TemperatureUnit) -> ((Double) -> Double) {
        if (targetUnit == self) {
            return {(value : Double) -> Double in
                return value}
        }
        return targetUnit.conversionFunctionFromCelsius() ∘ self.conversionFunctionToCelsius()
    }
    
    private func conversionFunctionFromCelsius() -> ConversionFunction {
        switch self {
            
        case .celsius: return {(value : Double) -> Double in
            return value}
            
        case .fahrenheit: return {(value : Double) -> Double in
            return value * 9.0/5.0 + 32.0}
            
        case .kelvin: return {(value : Double) -> Double in
            return value * 1.0 + 273.15}
        }
    }
    
    private   func conversionFunctionToCelsius() -> ConversionFunction {
        switch self {
        case .celsius: return {(value : Double) -> Double in
            return value}
            
        case .fahrenheit: return {(value : Double) -> Double in
            return value * 5.0/9.0 + (-32.0 * 5.0/9.0)}
            
        case .kelvin: return {(value : Double) -> Double in
            return value * 1.0 + (-273.15 * 1.0)}
        }
    }
    
    
}




struct Temperature {
    let value : Double
    let unit : TemperatureUnit
    
    func convert(toTargetUnit targetUnit: TemperatureUnit) -> Temperature {
        let convert = unit.conversionFunction(toUnit: targetUnit)
        return Temperature(value: convert(value), unit: targetUnit)
    }
}



extension Temperature : CustomStringConvertible {
    var description : String {
        get {
            return (NSString(format:"%.2d", self.value) as String) + self.unit.rawValue
        }
    }
}




