//
//  ViconDataStreamFramework.h
//  ViconDataStreamFramework
//
//  Created by Marty Pye on 31/03/14.
//  Copyright (c) 2014 Media Computing Group. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "VDSDelegate.h"

/**
 *  Enumeration for the different Axis directions. Can be up, down, left, right, forward or backward.
 */
typedef enum {
    Up,
    Down,
    Left,
    Right,
    Forward,
    Backward
} VDSAxisDirection;

@interface VDSClient : NSObject

+ (instancetype) sharedClient;

/**
 *  Starts the Vicon data stream.
 */
- (void) startStreaming;

/**
 *  Stops the Vicon data stream
 */
- (void) stopStreaming;

/**
 *  Sets whether to log additional data to the console. Should only be used for debugging.
 *
 *  @param shouldLogAllData if YES, then all system changes are logged to console. If NO,
 *  then only the necessary parameters are logged to the console.
 */
- (void) logAllData:(BOOL)shouldLogAllData;

/**
 *  Sets the axis mapping. It defines in which direction the x-, y- and z-axis should be pointing. Make sure the coordinate system 
 *  corresponds with the right-hand rule.
 *
 *  @param xDirection The direction in which the x-axis should point
 *  @param yDirection The direction in which the y-axis should point
 *  @param zDirection The direction in which the z-axis should point
 *
 *  @return Returns YES if the axes were set successfully. Returns NO if the axes couldn't be set with the given configuration.
 */
- (BOOL) setAxisMappingForX:(VDSAxisDirection) xDirection Y:(VDSAxisDirection) yDirection andZ:(VDSAxisDirection) zDirection;


/**
 *  The delegate for the vicon client. (Required)
 */
@property (weak) id <VDSDelegate> delegate;

/**
 *  This property should be set to YES if the marker data should be streamed, otherwise it should be set to NO.
 */
@property (readwrite, nonatomic, getter = isMarkerDataEnabled, setter = setMarkerDataEnabled:) BOOL markerDataEnabled;

/**
 *  This property should be set to YES if the segment data should be streamed, otherwise it should be set to NO.
 */
@property (readwrite, nonatomic, getter = isSegmentDataEnabled, setter = setSegmentDataEnabled:) BOOL segmentDataEnabled;

/**
 *  This property should be set to YES if the unlabled marker data should be streamed, otherwise it should be set to NO.
 */
@property (readwrite, nonatomic, getter = isUnlabeledMarkerDataEnabled, setter = setUnlabeledMarkerDataEnabled:) BOOL unlabeledMarkerDataEnabled; 

@end
