//
//  VDSSubject.h
//  ViconDataStreamFramework
//
//  Created by Marty Pye on 01/04/14.
//  Copyright (c) 2014 Media Computing Group. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "VDSSegment.h"

/**
 *  A VDSSubject is a subject, as defined in the Vicon Nexus Tracker. It contains information about all its segments and markers, which are stored in
 *  dictionaries. A VDSSubject also has a root segment, which is an ancestor of all other segments.
 */
@interface VDSSubject : NSObject

/**
 *  Creates a new subject.
 *
 *  @param theLabel    The name of the subject, as defined in the Vicon Nexus Tracker software.
 *  @param theSegments The dictionary of segments which belong to this subject. The segments in the dictionary can be accessed by name.
 *  @param theMarkers  The dictionary of markers which belong to this subject. The markers in the dictionary can be accessed by name.
 *
 *  @return The new subject.
 */
+ (VDSSubject*) subjectWithLabel:(NSString*)theLabel segments:(NSDictionary*)theSegments andMarkers:(NSDictionary*)theMarkers;

/**
 *  The dictionary of segments. The segments can be accessed by name, as defined in the Vicon Nexus Tracker.
 */
@property (copy, readonly) NSDictionary* segments;

/**
 *  The dictionary of markers. The markers can be accessed by name, as defined in the Vicon Nexus Tracker.
 */
@property (copy, readonly) NSDictionary* markers;

/**
 *  The name of this subject.
 */
@property (copy, readonly) NSString* label;

/**
 *  The x-coordinate of this subject, which is the x-coordinate of the origin marker of the root segment.
 */
@property (readonly, getter = x) double x;

/**
 *  The y-coordinate of this subject, which is the y-coordinate of the origin marker of the root segment.
 */
@property (readonly, getter = y) double y;

/**
 *  The z-coordinate of this subject, which is the z-coordinate of the origin marker of the root segment.
 */
@property (readonly, getter = z) double z;

/**
 *  The root segment of this subject, which is the common ancestor of all other segments.
 */
@property (readwrite) VDSSegment* rootSegment;


@end
