//
//  VDSSegment.h
//  ViconDataStreamFramework
//
//  Created by Marty Pye on 01/04/14.
//  Copyright (c) 2014 Media Computing Group. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "CATransform3D+Vector.h"
#import "VDSRotation.h"
/**
 *  A VDSSegment is a segment, as defined in the Vicon Nexus Tracker software. A segment's position in 3D space is the same position
 *  as that of its origin marker. A segment also knows its orientation in 3D-space, which is stored as a VDSRotation.
 */
@interface VDSSegment : NSObject

/**
 *  Creates a new segment.
 *
 *  @param theCoordinates The position of the segment.
 *  @param theName        The name of the segment as defined in the Vicon Nexus Tracker software.
 *
 *  @return The new segment.
 */
+ (VDSSegment *)segmentWithCoordinates:(CAVector3)theCoordinates andName:(NSString *)theName;

/**
 *  The name of the segment, as specified in the Vicon Nexus Tracker software.
 */
@property (copy, readonly) NSString *name;

/**
 *  The coordinates of the segment in 3D-space.
 */
@property (readonly) CAVector3 coordinates;

/**
 *  The x-coordinate of the segment in 3D-space
 */
@property (readonly, getter = x) double x;

/**
 *  The y-coordinate of the segment in 3D-space
 */
@property (readonly, getter = y) double y;

/**
 *  The z-coordinate of the segment in 3D-space
 */
@property (readonly, getter = z) double z;

/**
 *  The rotation information of the segment. Contains rotation details in different formats, such as Euler angles, Matrix etc.
 *  The rotation can be accessed in global coordinates or local coordinates (with respect to the parent segment).
 */
@property (nonatomic, strong) VDSRotation *rotation;

// backwards compatibility.
@property (readwrite) double eulerX __deprecated_msg("use rotation.localEulerX instead");
@property (readwrite) double eulerY __deprecated_msg("use rotation.localEulerX instead");
@property (readwrite) double eulerZ __deprecated_msg("use rotation.localEulerX instead");

/**
 *  If YES, then the segment is visible. If NO, then the segment is occluded and not visible for the cameras.
 */
@property (readwrite) BOOL visible;

@end
