/*jslint node: true, plusplus: true */
/*global setImmediate, clearImmediate*/
"use strict";

var helper = require("./helper");
var esprima = require("esprima");
var escodegen = require("escodegen");
var _ = require("underscore");
var domain = require("domain");

var variableNamePrefix = "___LIVE_CODING___";
var functionDeclarationIndexVarName = variableNamePrefix + "declarationIndex";
var functionDeclarationManagementObjectVarName = variableNamePrefix + "functionDeclarationManagementObject";
var errorDomainName = "executionDomain";

function getCallback(response) {
    // An array (that is used as a stack) for functions and loop counting variables
    var stack = [];

    return function (msg) {
        // console.log(stack);
        // console.log(msg);

        var stackItem = null;

        // Process the different message types before sending the message back to the client.
        // Entries of function, loops and try blocks are logged on the stack so if one of them gets exited
        // in an irregular way, the appropriate messages can be sent to the client.
        switch (msg.type) {
        case "function enter":
            // Put a new entry on the stack for the function
            stack.push({ type: "function", loc: msg.loc });
            break;
        case "function exit":
        case "return":
            // If the function was exited (possibly with a return), it's possible that some
            // loops were exited irregularily, so the loop exit messages have to be sent.
            // All stack entries for try-blocks are getting taken off the stack at this moment,
            // but all finaliziers are yet to be executed.
            while ((stackItem = stack.pop()).type !== "function") {
                if (stackItem.type !== "try") {
                    response({
                        type:           stackItem.type + " exit",
                        totalLoopCount: stackItem.count,
                        loc:            stackItem.loc
                    });
                }
            }
            break;
        case "begin try-catch":
            stack.push({ type: "try", loc: msg.loc });
            break;
        case "end try":
            // The top item on the stack should either be a try-block, in which case we take it off,
            // or something else (or empty stack), which means that an exception occurred and was handled
            if ((stack.length > 0) && (stack[stack.length - 1].type === "try")) {
                stack.pop();
            }
            break;
        case "throw":
            while ((stack.length > 0) && ((stackItem = stack.pop()).type !== "try")) {
                response({
                    type:           stackItem.type + " exit",
                    totalLoopCount: stackItem.count,
                    loc:            stackItem.loc
                });
            }

            // Put the try-block back on the stack for the catch message to know where to stop
            if (stackItem) {
                stack.push(stackItem);
            }
            break;
        case "catch":
            response({
                type: "caught exception",
                exception: msg.exception
            });
            while ((stackItem = stack.pop()).type !== "try") {
                response({
                    type:           stackItem.type + " exit",
                    totalLoopCount: stackItem.count,
                    loc:            stackItem.loc
                });
            }
            break;
        case "for-loop init":
        case "for-in-loop init":
        case "while-loop init":
            // Put a new entry on the stack for the loop, init at zero
            stack.push({ type: msg.type.slice(0, -5), loc: msg.loc, count: 0 });
            break;
        case "for-loop enter":
        case "for-in-loop enter":
        case "while-loop enter":
            // Increase the last number on the stack for every loop iteration
            stack[stack.length - 1].count += 1;
            break;
        case "for-loop exit":
        case "for-in-loop exit":
        case "while-loop exit":
            // Pop the last number from the stack (which should be the total number of iterations
            // if the innermost loop) and append it to the outgoing message
            msg.totalLoopCount = stack.pop().count;
            break;
        }

        response(msg);
    };
}


function getFunctionDeclarationManagementObject() {
    var functionDeclarationsLog = {};
    
    function getIdentifierForLocation(location) {
        return String(location.start.line) + "," + String(location.start.column) + ";" + String(location.end.line) + "," + String(location.end.column);
    }
    
    function incrementDeclarationsOfFunctionWithLocation(location) {
        var id = getIdentifierForLocation(location);
        functionDeclarationsLog[id] = (functionDeclarationsLog[id] === undefined) ? 1 : functionDeclarationsLog[id] + 1;
    }
    
    function getNumberOfDeclarationsOfFunctionWithLocation(location) {
        var id = getIdentifierForLocation(location);
        return functionDeclarationsLog[id] || 0;
    }
    
    return {
        incrementDeclarationsOfFunctionWithLocation: incrementDeclarationsOfFunctionWithLocation,
        getNumberOfDeclarationsOfFunctionWithLocation: getNumberOfDeclarationsOfFunctionWithLocation
    };
}

exports.Sandbox = function (response) {
    this.livecoding = { returnTmpVar: null };
    
    // Create the callback function
    var callback = getCallback(response);
    this.livecoding.callback = callback;
    
    // making the Date stringify correcly
    var jsonStringifyDate = Date.prototype.toJSON;
    this.livecoding.stringifyDate = function () {
        if (isNaN(this.getTime())) {
            return "Invalid Date";
        } // else
        return jsonStringifyDate.apply(this);
    }
    
    
    // support for async functions like setTimout and others
    
    function createDelayingFunctionWhichWrapsExecutionInFiberFromFunction(delayingFunction) {
        var wrappedDelayingFunction;
        if ((delayingFunction === setTimeout) || (delayingFunction === setInterval)) {
            wrappedDelayingFunction = function (functionToExecute, delay) {
                var forwardedArguments = _.toArray(arguments).slice(2); // remove the first two arguments, since we already use those
                return delayingFunction(function () {
                    
                    try {
                        functionToExecute.apply(undefined, forwardedArguments);
                    } catch (exception) {
                        callback({type: "uncaught exception", exception: exception});
                    }
                    
                }, delay);
            };
        } else if ((delayingFunction === setImmediate) || (delayingFunction === process.nextTick)) {
            wrappedDelayingFunction = function (functionToExecute) {
                var forwardedArguments = _.toArray(arguments).slice(1); // remove the first argument, since we already use that
                return delayingFunction(function () {
                    
                    try {
                        functionToExecute.apply(undefined, forwardedArguments);
                    } catch (exception) {
                        callback({type: "uncaught exception", exception: exception});
                    }
                    
                });
            };
        } else {
            throw new TypeError("Unrecognized delaying function.");
        }
        return wrappedDelayingFunction;
    }
    
    this.livecoding[errorDomainName] = domain.create();
            
    this.livecoding[errorDomainName].on("error", function (uncaughtException) {
        console.log("Execution Domain caught an exception: ", uncaughtException);
        console.log("Stack: ", helper.extractInterestingValuesFromStack(uncaughtException.stack));

        try {
            response({type: "uncaught exception", exception: uncaughtException});
        } catch (exception) {
            console.log("Could not send message to client, due to exception: ", exception);
            console.log("Stack: ", helper.extractInterestingValuesFromStack(exception.stack));
        }
    });

    
    // Pass the "process" and "require" objects to the sandbox
    this.process = process;
    this.require = require;
    
    // faking the define function that's used in many source files using require.js
    this.define = function (f) {
        if (!(typeof f === "function")) {
            return;
        }
        // just call the function we were given using the following fake parameters
        var require = function () {},
            exports = {},
            module;
        f(require, exports, module);
    };
    
    this.setTimeout = createDelayingFunctionWhichWrapsExecutionInFiberFromFunction(setTimeout);
    this.clearTimeout = clearTimeout;
    this.setInterval = createDelayingFunctionWhichWrapsExecutionInFiberFromFunction(setInterval);
    this.clearInterval = clearInterval;
    this.setImmediate = createDelayingFunctionWhichWrapsExecutionInFiberFromFunction(setImmediate);
    this.clearImmediate = clearImmediate;
    
    this.process = Object.create(process);
    this.process.nextTick = createDelayingFunctionWhichWrapsExecutionInFiberFromFunction(process.nextTick);
    
    // function declaration management stuff
    this[functionDeclarationManagementObjectVarName] = getFunctionDeclarationManagementObject();

    // Console resplacement for sandbox
    // TODO: handle other console properties
    function getLocationOfCaller() {
        // that's quite a hack. We are creating an error object, throwing it and catching it directly afterwards,
        // just to be able to use the call stack of the error to determine the location of the call two steps up the call stack
        // it's two steps because 
        // level 0 is were it was thrown, we know that, it will happen a few lines down from here
        // level 1 is the function that calls this function. It also knows where it calls this function so that's not interesting either.
        // level 2 is the one that calls the function that calls use, so that's the interesting one.
        var error = new Error();
        try {
            throw error;
        } catch (thrownError) {
            return {
                line: thrownError.stack[2].getLineNumber(),
                column: thrownError.stack[2].getColumnNumber()
            };
        }
        // if we reach this point, something strange went wrong, we should always reach the catch and leave the function there
        throw new Error("Should not be reachable");
    }
    this.console = {
        log: function () {
            var callLocation = getLocationOfCaller(),
                location = {
                    start: {line: callLocation.line, column: callLocation.column},
                    end: {line: callLocation.line, column: callLocation.column + 11} // since "console.log" is 11 characters long
                };
            
            callback({
                type:    "console log",
                values:   _.toArray(arguments),
                loc: location
            });
        },

        error: function () {
            var callLocation = getLocationOfCaller(),
                location = {
                    start: {line: callLocation.line, column: callLocation.column},
                    end: {line: callLocation.line, column: callLocation.column + 13} // since "console.error" is 13 characters long
                };
            callback({
                type:    "console error",
                values:   _.toArray(arguments),
                loc: location
            });
        }
    };
};

exports.instrumentCode = function (code, shouldGenerateSourceMap) {
    var tree, generatedCodeAndRawSourceMap, instrumentTree,
        parseTime, instrumentationTime, generationTime;

    //----- functions for different function declatations management stuff -----//
    function getFunctionDeclarationIndexVarNameForFunctionWithLocation(location) {
        return functionDeclarationIndexVarName + String(location.start.line) + "_" + String(location.start.column) + "__" + String(location.end.line) + "_" + String(location.end.column);
    }
    
    function getIncrementFunctionDeclarationCallForFunctionDeclarationOrExpression(functionDeclarationNode) {
        var stringForCallbackExpression = functionDeclarationManagementObjectVarName + ".incrementDeclarationsOfFunctionWithLocation("
                                            + JSON.stringify(functionDeclarationNode.loc) + ")",
            newParseError;
        try {
            return esprima.parse(stringForCallbackExpression).body[0];
        } catch (parseError) {
            console.log("Could not parse string for increment function declaration: ", stringForCallbackExpression);
            newParseError = new Error("Could not parse string: " + stringForCallbackExpression);
            newParseError.originalError = parseError;
            throw newParseError;
        }
    }
    
    function getDeclarationIndexVariableInitializationForFunctionDeclarationOrExpression(functionDeclarationNode) {
        var stringForCallbackExpression = "var " + getFunctionDeclarationIndexVarNameForFunctionWithLocation(functionDeclarationNode.loc) + " = "
                                        + functionDeclarationManagementObjectVarName + ".getNumberOfDeclarationsOfFunctionWithLocation("
                                        + JSON.stringify(functionDeclarationNode.loc) + ")",
            newParseError;
        try {
            return esprima.parse(stringForCallbackExpression).body[0];
        } catch (parseError) {
            console.log("Could not parse string for declaration index variable initialization: ", stringForCallbackExpression);
            newParseError = new Error("Could not parse string: " + stringForCallbackExpression);
            newParseError.originalError = parseError;
            throw newParseError;
        }
    }
    
    //----- Methods for different callback inserts -----//

    function getCallback(type, params, loc) {
        //console.log(type, params, loc);
        var stringForCallbackExpression = "global.livecoding.callback({ type: '" + type + "'"
                                        + (params ? ", " + params : "")
                                        + (loc ? ", loc: " + JSON.stringify(loc) : "")
                                        + " })";
        try {
            return esprima.parse(stringForCallbackExpression).body[0];
        } catch (parseError) {
            console.log("Could not parse string for general callback: ", stringForCallbackExpression);
            var newParseError = new Error("Could not parse string: " + stringForCallbackExpression);
            newParseError.originalError = parseError;
            throw newParseError;
        }
    }

    function getAssignmentCallback(name, loc) {
        // TODO: Make name parameter an AST-Node, instead of a string.
        // It will always be the left side of the assignment, but that can be a lot more than a simple string, which can lead 
        // to all kinds of parsing and generation issues when being sloppy like that.
        return getCallback("assignment", "name: '" + name.replace(/\'/g, "\\'") + "', value: " + name, loc);
    }

    function getUpdateCallback(name, loc) {
        return getCallback("update", "name: '" + name.replace(/\'/g, "\\'") + "', value: " + name, loc);
    }
    
    function getFunctionExpressionWithStatements(statements, loc) {
        // constructs a function expression AST node without parameters with the given statements in its body
        
        if (!(statements instanceof Array)) {
            throw new TypeError("statements parameters should be an Array.");
        } // else
        
        var functionExpression = {type: "FunctionExpression"};
        functionExpression.loc = loc; // we need to add the location so the source map works correctly
        functionExpression.id = null; // no name, anonymous function
        functionExpression.params = []; // no parameters to wrapper call
        functionExpression.defaults = [];
        functionExpression.body = {type: "BlockStatement"};
        functionExpression.body.loc = loc; // we need to add the location so the source map works correctly
        functionExpression.body.body = statements;
        functionExpression.rest = null;
        functionExpression.generator = false;
        functionExpression.expression = false;
        
        return functionExpression;
    }
    
    function getWrapperCallWithStatements(statements, loc) { // loc is the location the wrapper call should have, it's optional
        // this function takes an array of statements that should be wrapped in a function expression that's immediately called.
        // we construct the call expression and the function expression and use the statements as body of the function expression
        
        if (!(statements instanceof Array)) {
            throw new TypeError("statements parameters should be an Array.");
        } // else
        
        var wrapperCall = {}; // this node will contain the subtree that represents a call to an anonymous function that acts as described above
            
        wrapperCall.type = "CallExpression";
        wrapperCall.loc = loc; // we need to add the location so the source map works correctly
        wrapperCall.callee = getFunctionExpressionWithStatements(statements, loc);
        wrapperCall["arguments"] = []; // call without arguments
        
        return wrapperCall;
    }
    
    function getWrapperCallWithCallbackToWrapFunctionExpression(functionExpression) {
        // we want to replace the function expression with an anonymous functions that is immediately called 
        // and itself first calls the callback and then returns the original function expression.
        // this function constructs this wrapper call
        
        return getWrapperCallWithStatements([
            getDeclarationIndexVariableInitializationForFunctionDeclarationOrExpression(functionExpression),
            getIncrementFunctionDeclarationCallForFunctionDeclarationOrExpression(functionExpression),
            getCallback("function declaration", 
                        "declarationIndex: " + getFunctionDeclarationIndexVarNameForFunctionWithLocation(functionExpression.loc)
                            + ", bodyLoc: " + JSON.stringify(functionExpression.body.loc), 
                        functionExpression.loc),
            {type: "ReturnStatement", argument: functionExpression, loc: functionExpression.loc}
        ], functionExpression.loc); // we need to add the location so the source map works correctly
    }
    
    //----- Instrumentation functions -----//

    function instrumentFunction(func) {
        var name, params, body, parameterLocations;

        // Extract function name, parameters and location
        name = (func.id ? func.id.name : null);

        params = [];
        parameterLocations = [];
        func.params.forEach(function (value) {
            params.push("'" + value.name + "'");
            parameterLocations.push(value.loc);
        });

        if (func.body.type !== "BlockStatement") {
            // Function body is not a block, turn it into one
            body = func.body;
            func.body = {
                type:   "BlockStatement",
                body:   [body]
            };
        }

        func.body = instrumentTree(func.body);
        body = func.body.body;

        // Insert function enter callback into beginning of function body
        
        body.unshift(getCallback("function enter",
                                 "name: '" + name + "'"
                                     + ", declarationIndex: " + getFunctionDeclarationIndexVarNameForFunctionWithLocation(func.loc)
                                     + ", argNames: [" + params + "], argValues: arguments"
                                     + ", argLocs: " + JSON.stringify(parameterLocations), 
                                 func.loc)
                    );

        // Insert function exit callback at the end of the function body if the last statement isn't a return.
        // All the returns will be dealt with elsewhere.
        if (body[body.length - 1].type !== "ReturnStatement") {
            body.push(getCallback("function exit", "", func.loc));
        }
        
        return func;
    }

    function instrumentBlock(block) {
        var i, statementIndex, line, vars, arg, VAR_NAME = "global.livecoding.returnTmpVar";

        function getVarsInitCallback(declarations, loc) {
            var v = [];
            declarations.forEach(function (aDeclaration) {
                v.push("{ name: '" + aDeclaration.id.name + "', value: " + aDeclaration.id.name + ", loc: " + JSON.stringify(aDeclaration.loc) +  " }");
            });

            return getCallback("variables init", "vars: [" + v.join(", ") + "]", loc);
        }

        function getReturnCallback(value, loc) {
            return getCallback("return", "value: " + value, loc);
        }

        function getAssignment(name, valueNode) {
            var expression = { type: "ExpressionStatement"};
            expression.loc = valueNode.loc;
            expression.expression = { type: "AssignmentExpression" };
            expression.expression.loc = valueNode.loc;
            expression.expression.operator = "=";
            expression.expression.left = {
                type: "Identifier",
                name: name
            };
            expression.expression.right = valueNode;
            return expression;
        }

        function addAssignment(value) {
            if (value.init !== null) {
                line.init.expressions.push({
                    type:       "AssignmentExpression",
                    operator:   "=",
                    left:       value.id,
                    right:      value.init,
                    loc: value.loc
                });

                value.init = null;
            }
        }

        i = 0;
        while (i < block.length) {
            line = block[i];
            // console.log(line);

            // Instrument the for-loop declaration BEFORE passing it down further
            if (line.type === "ForStatement") {
                // Check if the initialization of the for-loop contains a variable declaration
                // and put it instead in front of the loop
                if (line.init && line.init.type === "VariableDeclaration") {
                    vars = line.init;
                    block.splice(i, 0, vars);

                    // Turn every declaration with a value into an assignment
                    line.init = {
                        type:           "SequenceExpression",
                        expressions:    []
                    };

                    vars.declarations.forEach(addAssignment);

                    if (line.init.length === 0) {
                        line.init = null;
                    }
                    i++;
                }

                // Insert the for-loop exit callback after the loop into the body
                block.splice(i + 1, 0, getCallback("for-loop exit", "", line.loc));
                i++;
            } else if (line.type === "ForInStatement") {
                // Check if the initialization of the for-in-loop contains a variable declaration
                // and put it instead in front of the loop
                if (line.left.type === "VariableDeclaration") {
                    vars = line.left;
                    block.splice(i, 0, vars);

                    // Turn the declaration into an identifier (we assume that the declaration had just one variable)
                    line.left = vars.declarations[0].id;

                    i++;
                }

                // Insert for-in-loop init and exit callbacks around the loop
                block.splice(i, 0, getCallback("for-in-loop init", "", line.loc));
                block.splice(i + 2, 0, getCallback("for-in-loop exit", "", line.loc));
                i += 2;
            }

            statementIndex = block.indexOf(line);
            if (statementIndex < 0) { throw new Error("Somehow the instrumentation went wrong. Cannot find line in block statement."); }
            block[statementIndex] = instrumentTree(line);

            switch (line.type) {
            case "VariableDeclaration":

                // Insert callback function call
                block.splice(i + 1, 0, getVarsInitCallback(line.declarations, line.loc));

                // Increment the counter, so that the inserted statement doesn't get instrumented itself
                i++;

                break;
            case "ExpressionStatement":
                if (line.expression.type === "AssignmentExpression") {
                    // Insert callback function call and increment the counter, but skip literal assignments
                    block.splice(i + 1, 0, getAssignmentCallback(escodegen.generate(line.expression.left), line.expression.loc));
                    i++;
                } else if (line.expression.type === "UpdateExpression") {
                    // Insert callback function call and increment the counter
                    block.splice(i + 1, 0, getUpdateCallback(escodegen.generate(line.expression.argument), line.expression.loc));
                    i++;
                }

                break;
            case "ReturnStatement":
                arg = line.argument;
                if (arg === null) {
                    // Empty return, insert callback
                    block.splice(i, 0, getCallback("function exit", "", line.loc));
                    i++;
                } else {
                    switch (arg.type) {
                    case "Identifier":
                        // Returning just a variable, insert callback BEFORE the return statement
                        block.splice(i, 0, getReturnCallback(arg.name, line.loc));
                        i++;
                        break;
                    case "Literal":
                        // Returning just a literal
                        block.splice(i, 0, getReturnCallback(arg.raw, line.loc));
                        i++;
                        break;
                    default:
                        // In all other cases we assume that there might be some calculations or function calls in the return statement.
                        // So, as to avoid side effects, we insert a surrogate variable, assign the original return value to it and return it.
                        block.splice(i, 0, getAssignment(VAR_NAME, arg));
                        block.splice(i + 1, 0, getReturnCallback(VAR_NAME, line.loc));
                        line.argument = { type: "Identifier", name: VAR_NAME };
                        i += 2;
                    }
                }
                break;
            case "WhileStatement":
            case "DoWhileStatement":
                // Insert while-loop enter and exit callbacks around the loop
                block.splice(i, 0, getCallback("while-loop init", "", line.loc));
                block.splice(i + 2, 0, getCallback("while-loop exit", "", line.loc));
                i += 2;
                break;
            case "TryStatement":
                block.splice(i + 1, 0, getCallback("end try", "", line.loc));
                i++;
                break;
            case "ThrowStatement":
                block.splice(i, 0, getCallback("throw", "", line.loc));
                i++;
                break;
            case "FunctionDeclaration":
                // insert function declaration callback and initialize the function's declarationIndex
                block.splice(i, 0,
                                    getDeclarationIndexVariableInitializationForFunctionDeclarationOrExpression(line),
                                    getIncrementFunctionDeclarationCallForFunctionDeclarationOrExpression(line),
                                    getCallback("function declaration", 
                                                "name: '" + line.id.name + "'" 
                                                    + ", declarationIndex: " + getFunctionDeclarationIndexVarNameForFunctionWithLocation(line.loc)
                                                    + ", bodyLoc: " + JSON.stringify(line.body.loc), 
                                                line.loc));
                i += 3;
                break;
            }

            i++;
        }
        
        return block;
    }

    function instrumentSequence(sequence) {
        var i = 0;
        while (i < sequence.length) {
            instrumentTree(sequence[i]);
            //console.log(sequence[i]);

            if (sequence[i].type === "AssignmentExpression") {
                // Add callback into sequence for assignments
                sequence.splice(i + 1, 0, getAssignmentCallback(escodegen.generate(sequence[i].left), sequence[i].loc).expression);
                i++;
            } else if (sequence[i].type === "UpdateExpression") {
                // Add callback into sequence for assignments
                sequence.splice(i + 1, 0, getUpdateCallback(escodegen.generate(sequence[i].argument), sequence[i].loc).expression);
                i++;
            }

            i++;
        }
        return sequence;
    }

    function instrumentCondition(condition, type) {
        // To instrument the test and capture possible assignments or function definitions,
        // we turn the test into a function that is declared and then immediatly called.
        // The test is passed as a parameter to the function and the result of the test
        // is the function's return value. The test is instrumented first to instrument
        // possible function entries or assignments.

        var wrapperCall = getWrapperCallWithStatements([
            getCallback(type, "value: result", condition.loc),
            {type: "ReturnStatement", argument: {type: "Identifier", name: "result", loc: condition.loc}, loc: condition.loc}
        ], condition.loc);
        
        wrapperCall["arguments"] = [condition]; // pass test as parameter to function
        wrapperCall.callee.params = [{type: "Identifier", name: "result", loc: condition.loc}]; // add a parameter named result to the function expression
        
        return wrapperCall;
    }

    function instrumentBooleanCondition(condition, type) {
        // To instrument the test and capture possible assignments or function definitions,
        // we turn the test into a function that is declared and then immediatly called.
        // The test is passed as a parameter to the function and the result of the test
        // is the function's return value. The test is instrumented first to instrument
        // possible function entries or assignments.

        if (condition) {
            var wrapperCall = getWrapperCallWithStatements([
                getCallback(type, "result: (result ? true : false), value: result", condition.loc),
                {type: "ReturnStatement", argument: {type: "Identifier", name: "result", loc: condition.loc}, loc: condition.loc}
            ], condition.loc);
            
            wrapperCall["arguments"] = [condition]; // pass test as parameter to function
            wrapperCall.callee.params = [{type: "Identifier", name: "result", loc: condition.loc}]; // add a parameter named result to the function expression
            
            return wrapperCall;
        } // else
        return condition;
    }

    function instrumentForLoop(loop) {
        function getForLoopInitCallback(loc) {
            return getCallback("for-loop init", "", loc).expression;
        }

        function instrumentInit(init) {
            if (init === null) {
                init = getForLoopInitCallback(loop.loc);
            } else {
                if (init.type !== "SequenceExpression") {
                    // Init part of for-loop is not a sequence, turn it into one, insert the callback and instrument the rest
                    init = {
                        type:           "SequenceExpression",
                        expressions:    [init]
                    };
                }

                init = instrumentTree(init);
                init.expressions.unshift(getForLoopInitCallback(loop.loc));
            }

            return init;
        }

        function instrumentUpdate(update) {
            if (update !== null) {
                if (update.type !== "SequenceExpression") {
                    // Update part of for-loop is not a sequence, turn it into one and instrument it
                    update = {
                        type:           "SequenceExpression",
                        expressions:    [update],
                        loc: update.loc
                    };
                }

                update = instrumentTree(update);
                
                
                // add update enter and exit callbacks
                update.expressions.unshift(getCallback("for-loop update enter", "", update.loc).expression);
                update.expressions.push(getCallback("for-loop update exit", "", update.loc).expression);
            }

            return update;
        }

        function instrumentBody(body) {
            if (body.type !== "BlockStatement") {
                // Body is not a block, turn it into block
                body = {
                    type:   "BlockStatement",
                    body:   [body]
                };
            }

            // Instrument body and insert callback
            body = instrumentTree(body);
            body.body.unshift(getCallback("for-loop enter", "", loop.loc));

            return body;
        }

        loop.init = instrumentInit(loop.init);
        loop.test = instrumentBooleanCondition(loop.test, "for-loop-test");
        loop.update = instrumentUpdate(loop.update);
        loop.body = instrumentBody(loop.body);
        
        return loop;
    }

    function instrumentForInLoop(loop) {
        var loopVar;

        loop.left = instrumentTree(loop.left);
        loop.right = instrumentTree(loop.right);

        if (loop.body.type !== "BlockStatement") {
            loop.body = {
                type:   "BlockStatement",
                body:   [loop.body]
            };
        }

        loop.body = instrumentTree(loop.body);

        loopVar = loop.left.name;
        loop.body.body.unshift(getCallback("for-in-loop enter", "loopVar: '" + loopVar.replace(/\'/g, "\\'") + "', value: " + loopVar + ", loopVarLoc: " + JSON.stringify(loop.left.loc), loop.loc));
        
        return loop;
    }

    function instrumentWhileLoop(loop) {
        loop.test = instrumentBooleanCondition(loop.test, "while-loop-test");

        if (loop.body.type !== "BlockStatement") {
            loop.body = {
                type:   "BlockStatement",
                body:   [loop.body]
            };
        }
        loop.body = instrumentTree(loop.body);

        loop.body.body.unshift(getCallback("while-loop enter", "", loop.loc));
        
        return loop;
    }

    function instrumentIfStatement(ifSt) {
        ifSt.test = instrumentBooleanCondition(ifSt.test, "if-test");

        // Turn the branches of the if-statement into blocks, if they aren't already, and instrument them

        if (ifSt.consequent.type !== "BlockStatement") {
            ifSt.consequent = {
                type: "BlockStatement",
                body: [ifSt.consequent]
            };
        }
        ifSt.consequent = instrumentTree(ifSt.consequent);

        if (ifSt.alternate && ifSt.alternate.type !== "BlockStatement") {
            ifSt.alternate = {
                type: "BlockStatement",
                body: [ifSt.alternate]
            };
        }
        ifSt.alternate = instrumentTree(ifSt.alternate);
        
        return ifSt;
    }

    function instrumentTryStatement(trySt) {
        var type = "begin try";
        trySt.block = instrumentTree(trySt.block);

        // We add to the message whether the try-block contains a handler (catch-part)
        // If not, it has to contain a finally-part
        if (trySt.handlers.length > 0) {
            type += "-catch";
            trySt.handlers = trySt.handlers.map(instrumentTree);
        } else {
            type += "-finally";
        }

        trySt.finalizer = instrumentTree(trySt.finalizer);

        trySt.block.body.unshift(getCallback(type, "", trySt.loc));
        
        return trySt;
    }

    
    instrumentTree = function (tree) {
        // if (tree) console.log(tree.loc);

        // TODO handle exit from labelled statements
        if (tree) {
            switch (tree.type) {
            case "Program":
            case "BlockStatement":
                tree.body = instrumentBlock(tree.body);
                break;
            case "SequenceExpression":
                tree.expressions = instrumentSequence(tree.expressions);
                break;
            case "VariableDeclaration":
                tree.declarations = tree.declarations.map(instrumentTree);
                break;
            case "VariableDeclarator":
                tree.init = instrumentTree(tree.init);
                break;
            case "FunctionDeclaration":
                tree = instrumentFunction(tree);
                break;
            case "FunctionExpression":
                tree = getWrapperCallWithCallbackToWrapFunctionExpression(instrumentFunction(tree));
                break;
            case "ExpressionStatement":
                tree.expression = instrumentTree(tree.expression);
                break;
            case "ArrayExpression":
                tree.elements = tree.elements.map(instrumentTree);
                break;
            case "CallExpression":
            case "NewExpression":
                tree.callee = instrumentTree(tree.callee);
                tree["arguments"] = tree["arguments"].map(instrumentTree);
                break;
            case "MemberExpression":
                tree.object = instrumentTree(tree.object);
                tree.property = instrumentTree(tree.property);
                break;
            case "AssignmentExpression":
            case "BinaryExpression":
            case "LogicalExpression":
                tree.left = instrumentTree(tree.left);
                tree.right = instrumentTree(tree.right);
                break;
            case "ObjectExpression":
                tree.properties = tree.properties.map(instrumentTree);
                break;
            case "Property":
                tree.key = instrumentTree(tree.key);
                tree.value = instrumentTree(tree.value);
                break;
            case "IfStatement":
                tree = instrumentIfStatement(tree);
                break;
            case "SwitchStatement":
                tree.discriminant = instrumentCondition(tree.discriminant, "switch-condition");
                tree.cases = tree.cases.map(instrumentTree);
                break;
            case "SwitchCase":
                tree.test = instrumentTree(tree.test);
                tree.consequent = instrumentBlock(tree.consequent);
                break;
            case "ConditionalExpression":
                tree.test = instrumentTree(tree.test);
                tree.consequent = instrumentTree(tree.consequent);
                tree.alternate = instrumentTree(tree.alternate);
                break;
            case "WhileStatement":
            case "DoWhileStatement":
                tree = instrumentWhileLoop(tree);
                break;
            case "ForStatement":
                tree = instrumentForLoop(tree);
                break;
            case "ForInStatement":
                tree = instrumentForInLoop(tree);
                break;
            case "TryStatement":
                tree = instrumentTryStatement(tree);
                break;
            case "CatchClause":
                tree.body = instrumentTree(tree.body);
                tree.body.body.unshift(getCallback("catch", "exception: " + tree.param.name, tree.loc));
                break;
            case "LabeledStatement":
                tree.body = instrumentTree(tree.body);
                break;
            case "UpdateExpression":
            case "ReturnStatement":
            case "ThrowStatement":
            case "UnaryExpression":
                tree.argument = instrumentTree(tree.argument);
                break;
            case "Literal":
            case "Identifier":
            case "ThisExpression":
            case "EmptyStatement":
            case "BreakStatement":
            case "ContinueStatement":
                break;
            default:
                console.error("Error: uncaught case during instrumentation of type " + tree.type +
                    (tree.loc ? " in lines " + tree.loc.start.line + "-" + tree.loc.end.line : ""));
            }
        }
        
        return tree;
    };

    function wrapStatementsInTryCatch(statements, loc) { // second parameter: location of code, optional, will be added to outer blocks to keep source map working
        if (!(statements instanceof Array)) {
            throw new TypeError("statements parameters should be an Array.");
        } // else
        
        var tryCatchBlock = {type: "TryStatement"},
            catchClause = {type: "CatchClause"},
            callback = {type: "CallExpression"},
            paramObject = {type: "ObjectExpression"},
            exceptionVarName = variableNamePrefix + "executionError";
        tryCatchBlock.loc = loc; // use the given location
        tryCatchBlock.block = {type: "BlockStatement"};
        tryCatchBlock.block.loc = loc;
        tryCatchBlock.block.body = statements; // the original statements should be in the try block
        tryCatchBlock.guardedHandlers = []; // whatever
        tryCatchBlock.handlers = [catchClause];
        tryCatchBlock.finalizer = null;
        
        catchClause.param = {
            type: "Identifier",
            name: exceptionVarName
        };
        catchClause.body = {
            type: "BlockStatement",
            body: [{type: "ExpressionStatement", expression: callback}]
        };
        
        callback.callee = {
            type: "MemberExpression",
            object: {
                type: "MemberExpression",
                object: {type: "Identifier", name: "global"},
                property: {type: "Identifier", name: "livecoding"}
            },
            property: {type: "Identifier", name: "callback"}
        };
        callback["arguments"] = [paramObject];
        
        paramObject.properties = [
            {
                key: {type: "Identifier", name: "type"},
                value: {type: "Literal", value: "uncaught exception", raw: "'uncaught exception'"},
                kind: "init"
            },
            {
                key: {type: "Identifier", name: "exception"},
                value: { type: "Identifier", name: exceptionVarName},
                kind: "init"
            }
        ];
        
        return tryCatchBlock;
    }
    
    
    if (typeof code === "string") {
        // Parse the code and instrument the syntax tree
        parseTime = helper.timeExecution(function () {
            tree = esprima.parse(code, { loc: true, raw: true });
        });
        instrumentationTime = helper.timeExecution(function () {
            instrumentTree(tree);
        
            // Wrap the whole program in a try-catch block to get uncaught exceptions
            tree.body = [wrapStatementsInTryCatch(tree.body, tree.loc)];
        });
        
        generationTime = helper.timeExecution(function () {
            if (shouldGenerateSourceMap) {
                generatedCodeAndRawSourceMap = escodegen.generate(tree, {sourceMap: "myFile.js", sourceMapWithCode: true});
            } else {
                generatedCodeAndRawSourceMap = {};
                generatedCodeAndRawSourceMap.code = escodegen.generate(tree);
            }
        });
    } else { console.error("Couldn\'t instrument code, variable is not a string."); }

    return {code: generatedCodeAndRawSourceMap.code,
            rawSourceMap: generatedCodeAndRawSourceMap.map && String(generatedCodeAndRawSourceMap.map),
            durations: { parseTime: parseTime,
                         instrumentationTime: instrumentationTime,
                         generationTime: generationTime
                       }
           };
};
