/*jslint node: true, plusplus: true */
"use strict";

process.title = "LiveCoding-node server";

var ws = require("ws");
var fork = require("child_process").fork;

var PORT = 1234,
    SHARED_CONSTANTS = require("./sharedConstants");

// Server Session
function ServerSession(socket) {
    this.socket = socket;
    this.socket.on("message", this.onMessage.bind(this));
    this.socket.on("close", this.onClose.bind(this));

    // Map of all runtimes for this session
    this.runtimes = {};
}

ServerSession.prototype.onMessage = function (message) {
    var response, self = this;

    function sendResponse(response) {
        self.send((response.id ? response.id + "#" : "") + JSON.stringify(response));
    }

    response = this.handleMessage(message);
    if (response !== undefined) {
        if (response && typeof response.then === "function") {
            response.then(sendResponse);
        } else {
            sendResponse(response);
        }
    }
};

ServerSession.prototype.onClose = function () {
    console.log("Closing session");
    this.stopRuntimes();
};

// Stop the runtime with a particular id
ServerSession.prototype.stopRuntime = function (id) {
    // Stop all runtimes for this session
    var executionManager;
    if (this.runtimes.hasOwnProperty(id)) {
        executionManager = this.runtimes[id];
        try {
            executionManager.isBroken = true; // ignore everything the process sends now
            console.log("Killing child with process id: ", executionManager.pid);
            executionManager.disconnect(); // the execution manager will notice the disconnect, kill its children and exit
            delete this.runtimes[id];
            return;
        } catch (killError) {
            console.log("Could not kill old runtime due to error: ", killError);
        }
    }
};

// Stops all runtimes and resets the runtimes object
ServerSession.prototype.stopRuntimes = function () {
    // Stop all runtimes for this session
    var runtimeID, runtime;
    for (runtimeID in this.runtimes) {
        if (this.runtimes.hasOwnProperty(runtimeID)) {
            this.stopRuntime(runtimeID);
        }
    }
    this.runtimes = {};
};

ServerSession.prototype.send = function (message) {
    this.socket.send(message, function (error) {
        if (error) {
            console.log("Sending failed with error:", error);
        }
    });
};

ServerSession.prototype.handleMessage = function (message) {
    var
        self = this,
        result,
        options;            // Options for the runtime (embedded in the message or initialized here)

    function executeCode(code, id, options, response) {
        if (options.stopOldRuntimes === true) {
            self.stopRuntimes();
        }

        var executionManager = fork("./executionManager");
        self.runtimes[id] = executionManager;

        executionManager.on("message", function (msg) {
            if (executionManager.isBroken) {
                // yeah, whatever. We already killed it, ignore the messages.
                return;
            }
            switch (msg.type) {
            case "log":
            case "error":
            case "message array":
                msg.id = id;
                response(msg);
                break;
            default:
                response({ id: id, type: "error", message: "Unknown message type for message: " + JSON.stringify(msg)});
            }
        });
        executionManager.on("exit", function (code, signal) {
            if (code === 0) {
                // seems to have exited normally
                console.log("Child process exited.");
                return;
            } // else
            if (executionManager.isBroken) {
                // ignore whatever problems the process has. We already killed it.
                return;
            } // else
            
            if (code === SHARED_CONSTANTS.SEND_ERROR_CODE) {
                console.log("Child exited due to send error.");
                return;
            }
            
            console.log("Child process exited with code: " + code + " after receiving signal: " + signal);
            if (code !== 0) {
                // some error occured
                response({
                    id: id,
                    type: "message array",
                    messages: [{
                        type: "executionError",
                        error: {message: "Process exited abnormally."}
                    }]
                });
            }
        });
        executionManager.on("disconnect", function () {
            console.log("Child disconnected...");
        });
        executionManager.on("error", function (error) {
            if (!executionManager.isBroken) { // if it's broken it means we killed it already, ignore errors in that case
                console.log("Child process encountered error: ", error);
            }
        });

        executionManager.send({ type: "code", code: code, options: options });
    }

    try {
        message = JSON.parse(message);
    } catch (e) { this.send({ type: "error", message: "Error: Message is not a JSON string." }); }

    if (message && message.type) {
        switch (message.type) {
        case "code":
            if (message.code && typeof message.code === "string") {
                // Check if the message has an id and it is formed correctly (i.e., it should not contain a #)
                if (!message.id) {
                    this.send({ type: "error", message: "Error: Message must contain an id." });
                } else if ((typeof message.id !== "string") && (typeof message.id !== "number")) {
                    this.send({ type: "error", message: "Error: The id has to be a number or a string without a # sign." });
                } else if ((typeof message.id === "string") && (message.id.indexOf("#") >= 0)) {
                    this.send({ type: "error", message: "Error: Id can't contain a #." });
                } else {
                    // Check the options and initialize if necessary
                    if (message.options) {
                        options = message.options;
                    } else {
                        options = {};
                    }

                    if (options.disposeAfterExit === undefined) {
                        options.disposeAfterExit = true;
                    }

                    if (options.stopOldRuntimes === undefined) {
                        options.stopOldRuntimes = true;
                    }

                    // A piece of code was passed directly, try to execute it in the sandbox
                    result = {};
                    result.then = function (callback) {
                        executeCode(message.code, message.id, options, callback);
                    };
                }
            } else { this.send({ type: "error", message: "Error: Message type is 'code', but there is no code attached." }); }

            break;
        case "stopRuntime":
            if (message.runtimeID) {
                this.stopRuntime(message.runtimeID);
            }

            break;
        default:
            this.send({ type: "error", message: "Error: Unknown message type." });
            break;
        }
    }

    return result;
};

// Server (WebSocket)
function Server(options) {
    this.wss = new ws.Server(options);
    this.wss.on("connection", this.onConnect.bind(this));
    this.wss.on("error", function (error) {
        console.log("Websocket Server Error: ", error);
    });
    this.sessions = [];
}

Server.prototype.onConnect = function (socket) {
    console.log("Client connected");

    var session = new ServerSession(socket);
    this.configureSession(session);
    socket.on("close", this.onClose.bind(this, session));
    socket.on("error", function (error) {
        console.log("error on socket: ", error);
    });
    this.sessions.push(session);
};

Server.prototype.onClose = function (session) {
    console.log("Client disconnected");

    var sessionIndex = this.sessions.indexOf(session);
    if (sessionIndex > -1) {
        session.stopRuntimes();
        delete this.sessions[sessionIndex];
    }
};

Server.prototype.configureSession = function (session) {};

process.on("uncaught exception", function (exception) {
   console.log("Uncaught exception in server process. This is not supposed to happen! ", exception);
});


console.log("Starting server on port " + PORT);
var server = new Server({ port : PORT });