/*jslint node: true, plusplus: true, nomen: true */
'use strict';

var helper = require("./helper"),
    utils = require("./utils");
var Contextify = require("contextify"),
    _ = require("underscore"),
    fork = require("child_process").fork,
    domain = require("domain"),
    jQuery = require("jquery");
require("./cycle.js");
    
var specialStrings = {
    functionPlaceholder : "__LIVECODING__FUNCTION",
    undefinedPlaceholder : "__LIVECODING__UNDEFINED",
    specialNumbersPlaceholder: {
        positiveInfinity: "__LIVECODING__POSITIVE_INFINITY",
        negativeInfinity: "__LIVECODING__NEGATIVE_INFINITY",
        notANumber: "__LIVECODING__NaN"
    },
    jQueryType : "__LIVECODING__JQUERY",
    htmlElementType: "__LIVECODING__HTML",
    dateType: "__LIVECODING__DATE"
},
    SANDBOXED_CODE_FILE_NAME = "_LIVE_CODING_SANDBOX_",
    MAX_MESSAGE_BUFFER_SIZE = 1000,
    MAX_WAIT_TIME_BETWEEN_MESSAGE_SENDS = 100,
    SHARED_CONSTANTS = require("./sharedConstants"),
    MESSAGE_PROPERTIES_TO_DECYCLE = ["value", "exception"],
    MESSAGE_ARRAY_PROPERTIES_TO_DECYCLE = ["argValues", "values"],
    MESSAGE_ARRAYS_WITH_PROPERTIES_TO_DECYCLE = [{arrayKey: "vars", propertyKey: "value"}];


// make location information in errors more easily accessible, by forcing the stack to be an array of call-items, not a string
Error.prepareStackTrace = function (err, stack) { return stack; };


var messageStringsToSend = [];


// setting up robust sending with error catching
var sendErrorDomain = domain.create();
sendErrorDomain.on("error", function(sendError) {
    console.log("Domain caught send error with message: ", sendError.message);
    console.log("Stack: ", helper.extractInterestingValuesFromStack(sendError.stack));
    console.log("Exiting process due to send error...");
    process.exit(SHARED_CONSTANTS.SEND_ERROR_CODE);
});

function send(message) {
    try {
        sendErrorDomain.run(function () {
            process.send(message);
        });
    } catch (sendError) {
        console.log("Error sending message: ", message);
        console.log("Error.message: ", sendError.message);
        console.log("Stack: ", helper.extractInterestingValuesFromStack(sendError.stack));
        console.log("Exiting process due to send error...");
        process.exit(SHARED_CONSTANTS.SEND_ERROR_CODE);
    }
}


function tryToStringifyMessage(msg) {
    function JSONReplacer(key, value) {
        if (key === undefined) {
            return value; // should only happen for the root object
        }
        if ((value === undefined) && (key !== "loc")) {
            return specialStrings.undefinedPlaceholder;
        }
        
        if (typeof value === "number") {
            // check for some special cases that are not correctly stringified by JSON
            switch (value) {
            case Infinity:
                return specialStrings.specialNumbersPlaceholder.positiveInfinity;
            case -Infinity:
                return specialStrings.specialNumbersPlaceholder.negativeInfinity;
            }
            if (isNaN(value)) {
                return specialStrings.specialNumbersPlaceholder.notANumber;
            }
        }
        
        // special case for jquery objects
        if (value instanceof jQuery) {
            try {
                if (value.length === 0) {
                    return {type: specialStrings.jQueryType, value: []};
                } // else
                var transformedValues = [];
                value.each(function (index, element) {
                    var html = element.outerHTML;
                    if (html) {
                        transformedValues.push(html);
                    } else {
                        transformedValues.push(element);
                    }
                
                });
                return {type: specialStrings.jQueryType, value: transformedValues};
            } catch (jqueryTypeError) {
                // well that didnt work out. Just handle it as usual.
            }
        }
        
        // special case for HTML stuff
        if (value && value.outerHTML) {
            return {type: specialStrings.htmlElementType, value: value.outerHTML};
        }
        
        if (typeof value === "function") {
            return specialStrings.functionPlaceholder;
        }
        if ((key === "argValues")) {
            try {
                 // the arguments object is not an actual array, so JSON would stringify
                // it as an object. But we want to have it as an array.
                return _.toArray(value);
            } catch (e) {
                // however, it might be the case that some other object in the message object tree
                // has a property "argValues", in that case, it might not be something that map can be applied to
                // if it somehow doesn't work, we just return the original value
                console.log("Could not convert value to array: ", e);
                return value;
            }
        }
        return value;
    }
    
    var stringifiedMessage, decycledMessage;
    try {
        stringifiedMessage = JSON.stringify(msg, JSONReplacer);
    } catch (stringifyError) { // could be due to a circular object
        // circular objects should only appear in some of the properties of the message, so we only decycle these
        decycledMessage = msg;
        MESSAGE_PROPERTIES_TO_DECYCLE.forEach(function (propertyToDecycle) {
            if (decycledMessage.hasOwnProperty(propertyToDecycle)) {
                decycledMessage[propertyToDecycle] = JSON.decycle(decycledMessage[propertyToDecycle]);
            }
        });
        MESSAGE_ARRAY_PROPERTIES_TO_DECYCLE.forEach(function (propertyToDecycle) {
            // these are properties that are arrays for which we only want to decycle the indivudal objects in the array
            if (decycledMessage.hasOwnProperty(propertyToDecycle)) {
                decycledMessage[propertyToDecycle] = _.map(decycledMessage[propertyToDecycle], JSON.decycle);
            }
        });
        MESSAGE_ARRAYS_WITH_PROPERTIES_TO_DECYCLE.forEach(function (arrayAndPropertyKeyObject) {
            // these are properties that are arrays for which we only want to decycle the indivudal objects in the array
            var arrayKey = arrayAndPropertyKeyObject.arrayKey,
                propertyKey = arrayAndPropertyKeyObject.propertyKey;
            if (decycledMessage.hasOwnProperty(arrayKey)) {
                _.forEach(decycledMessage[arrayKey], function (element) {
                    if (element.hasOwnProperty(propertyKey)) {
                        element[propertyKey] = JSON.decycle(element[propertyKey]);
                    }
                });
            }
        });
        stringifiedMessage = JSON.stringify(decycledMessage, JSONReplacer);
    }
    return stringifiedMessage;
}

function extractDataOfExceptionInMessageToSupportStringification(message) {
    if ((message.type === "uncaught exception")
            || (message.type === "caught exception")
            || (message.type === "catch")) {
        var exception = message.exception,
            stringifiableException = {message: String(message.exception)},
            errorLocation,
            stackFramesInExecutedCode;
//            if (message.type === "uncaught exception") {
//                console.log("Uncaught exception: ", exception);
//                console.log("Stack: ", helper.extractInterestingValuesFromStack(exception.stack));
//            }
        stackFramesInExecutedCode = exception.stack && exception.stack.filter && exception.stack.filter(
            function (aStackFrame) {
                return (typeof aStackFrame.getFileName === "function") && (aStackFrame.getFileName() === SANDBOXED_CODE_FILE_NAME);
            }
        );
        if (stackFramesInExecutedCode && stackFramesInExecutedCode[0]) { // check whether we have one in our code
            // and simply take the first one, since that should be the topmost one, the one calling the throwing code
            if (typeof stackFramesInExecutedCode[0].getLineNumber === "function") { // take the first one
                errorLocation = {
                    line: stackFramesInExecutedCode[0].getLineNumber(),
                    column: stackFramesInExecutedCode[0].getColumnNumber() - 1
                };
                if (typeof errorLocation.line !== "number") {
                    errorLocation = undefined;
                }
            }
        }
        if (exception.message) {
            stringifiableException = {
                message: exception.message,
                type: exception.type
            };
        }
        message.exception = stringifiableException;
        message.stack = exception.stack && exception.stack.map(String);
        if (message.type !== "catch") {
            // catch messages should keep the location of the catch block
            // others take the location of the error occurence itself
            message.loc = errorLocation && {start: errorLocation, end: errorLocation};
        }
    }
}

function executeCode(code, options) {
    
    var sandbox, t, generatedCodeAndTimingInfo, execution,
        executionDomain,
        lastMessageSentTimestamp;
    
    
    function sendBufferedMessages() {
        if (messageStringsToSend.length > 0) {
            lastMessageSentTimestamp = Date.now(); // used to now when it is time to send the next bunch of messages
            try {
                send({ type: "message array", stringifiedMessages: messageStringsToSend});
            } catch (sendException) {
                
                var messages = messageStringsToSend.map(JSON.parse);
                console.log("Could not send messages to parent due to error: ", sendException);
                console.log("Stack: ", helper.extractInterestingValuesFromStack(sendException.stack));
                console.log("Buffered messages: ", messages);
            }
            messageStringsToSend = []; // empty the array
        }
    }
    
    var timerToSendMessages;
    function scheduleMessageSending() {
        if (!timerToSendMessages) {
            timerToSendMessages = setTimeout(function () {
                timerToSendMessages = undefined;
                sendBufferedMessages();
            }, MAX_WAIT_TIME_BETWEEN_MESSAGE_SENDS);
        }
    }
    
    function response(msg) {
        extractDataOfExceptionInMessageToSupportStringification(msg);
        
        try {
            messageStringsToSend.push(tryToStringifyMessage(msg));
            scheduleMessageSending();
        
            if (msg.type === "uncaught exception") {
                // we are in an unknown state now. Be careful and try to send all messages now, we never know what happens next.
                sendBufferedMessages();
            }
            
            if ((messageStringsToSend.length >= MAX_MESSAGE_BUFFER_SIZE)
                    || ((Date.now() - lastMessageSentTimestamp) >= MAX_WAIT_TIME_BETWEEN_MESSAGE_SENDS)) {
                
                sendBufferedMessages();
            }
        } catch (e) {
            console.log("Error during callback with message: ", msg);
            console.log("Error: ", e);
            console.log("Stack: ", helper.extractInterestingValuesFromStack(e.stack));
            try {
                send({ type: "error", message: "Error during callback: " + e, stack: helper.extractInterestingValuesFromStack(e.stack) });
            } catch (sendException) {
                // if that doesn't work it might just be that the connection to the parent is already broken, 
                // ignore it
            }
        }
    }
    
    

    
    
    
    try {
        generatedCodeAndTimingInfo = utils.instrumentCode(code);
        code = generatedCodeAndTimingInfo.code;
        //console.log("Code successfully instrumented in: ", generatedCodeAndTimingInfo.durations);
        send({ type: "log", message: "Code successfully instrumented in " + (generatedCodeAndTimingInfo.durations.parseTime + generatedCodeAndTimingInfo.durations.instrumentationTime + (generatedCodeAndTimingInfo.durations.generationTime || 0)) + " ms." });
        
        //console.log("instrumented code: \n", code);

        
        try {
            executionDomain = domain.create();
            
            executionDomain.on("error", function (uncaughtException) {
                console.log("Domain caught an exception with message: ", uncaughtException.message);
                console.log("Stack: ", helper.extractInterestingValuesFromStack(uncaughtException.stack));

                try {
                    response({type: "uncaught exception", exception: uncaughtException});
                } catch (exception) {
                    console.log("Could not send message to client, due to exception: ", exception);
                    console.log("Stack: ", helper.extractInterestingValuesFromStack(uncaughtException.stack));
                    console.log("Messages left to send: ", messageStringsToSend);
                }
            });
            
            executionDomain.run(function () {
                sandbox = Contextify(new utils.Sandbox(response));
                sandbox.global = sandbox.getGlobal();
    
                // we make sure that the stack trace of the error object is not created as a string but preserved as collection of objects
                // by modyfing the Error objects in the Sandbox/VM
                sandbox.run("Error.prepareStackTrace = function (err, stack) { return stack; };");
                // overwriting the toJSON function of date, because we want to now that it was originally a date
                // we need to override the toJSON function of Date, and we can only do it once the sandbox is established (so there is a Date object)
                // This is necessary because JSON seems to prefer the toJSON method over any custom replacer method given to a stringfy()-call
                // We rely on the sandbox having another function that can stringify dates and use that for the actual stringification.
                // Other than that we use the same format as in the JSONReplacer at the beginning of the file.
                // And yes: That's a hell of a hack. I couldn't find a better way. I really tried. I'm sorry...
                sandbox.run("Date.prototype.toJSON = function () { return {type: '" + specialStrings.dateType + "', value: global.livecoding.stringifyDate.apply(this)};};");
        
                t = helper.timeExecution(function () {
                    lastMessageSentTimestamp = Date.now(); // save to know when it is time to send messages
                    sandbox.run(code, SANDBOXED_CODE_FILE_NAME);
                    sendBufferedMessages(); // send all messages that are left
                });
    
                if (options.disposeAfterExit === true) {
                    console.log("Disposing Sandbox!");
                    sandbox.dispose();
                }
    
                send({ type: "log", message: "Code successfully executed in " + t + " ms." });
            });
        } catch (e) {
            console.log("Error executing code: ", e, "stack: ", helper.extractInterestingValuesFromStack(e.stack));
            
            // If some message have already been collected, send them off BEFORE the exception
            sendBufferedMessages();
            
            send({ type: "error", message: "Error executing code: " + e, stack: helper.extractInterestingValuesFromStack(e.stack) });
        }
    } catch (instrumentationError) {
        console.log("Instrumentation Error: ", instrumentationError);
        console.log("Stack: ", helper.extractInterestingValuesFromStack(instrumentationError.stack));
        //console.log("Code: " + code);
        var exceptionString = tryToStringifyMessage(instrumentationError);
        if (exceptionString) {
            send({ type: "error", message: "Error instrumenting code: " + exceptionString });
        } else {
            exceptionString = tryToStringifyMessage(instrumentationError.message);
            if (exceptionString) {
                send({ type: "error", message: "Error instrumenting code (not stringifiable). Message: " + exceptionString });
            } else {
                send({ type: "error", message: "Error instrumenting code (not stringifiable)." + instrumentationError });
            }
        }
    }
}

process.once("message", function (msg) {
    if (msg.type === "code") {
        executeCode(msg.code, msg.options);
    }
});

process.on("uncaught exception", function (exception) {
    console.log("An uncaught exception occured in child process and was caught by process: ", exception);
    console.log("Stack: ", helper.extractInterestingValuesFromStack(exception.stack));
});

process.on("exit", function() {
    if (messageStringsToSend.length > 0) {
        console.log("Execution process is exciting although there are messages left: " + messageStringsToSend.length);
        send({type: "error", message: "Execution process is exciting although there are messages left: " + messageStringsToSend.length});
    }
});

send({ type: "log", message: "Starting runtime" });
