/*jslint node: true, vars: true, nomen: true */
"use strict";

process.title = "LiveCoding-node execution manager";

var helper = require("./helper"),
    utils = require("./utils"),
    SourceMapConsumer = require("source-map").SourceMapConsumer,
    fork = require("child_process").fork,
    domain = require("domain");

var WAIT_TIME_BETWEEN_MESSAGEPROCESSING = 10,
    SHARED_CONSTANTS = require("./sharedConstants");

var sourceMap,
    runtime,
    sourceMapGenerator,
    messagesToCorrect = [],
    correctedMessages = [];


// setting up robust sending with error catching
var sendErrorDomain = domain.create();
sendErrorDomain.on("error", function(sendError) {
    console.log("Domain caught send error with message: ", sendError.message);
    console.log("Stack: ", helper.extractInterestingValuesFromStack(sendError.stack));
    console.log("Exiting process due to send error...");
    process.exit(SHARED_CONSTANTS.SEND_ERROR_CODE);
});

function sendToParent(message) {
    try {
        sendErrorDomain.run(function () {
            process.send(message);
        });
    } catch (sendError) {
        console.log("Error sending message: ", message);
        console.log("Error.message: ", sendError.message);
        console.log("Stack: ", helper.extractInterestingValuesFromStack(sendError.stack));
        console.log("Exiting process due to send error...");
        process.exit(SHARED_CONSTANTS.SEND_ERROR_CODE);
    }
}




// startup stuff

function startSourceMapGenerator(code) {
    sourceMapGenerator = fork("./sourceMapGenerator");
    
    sourceMapGenerator.on("message", function (msg) {
        if (msg.type === "sourceMap") {
            sourceMap = new SourceMapConsumer(msg.sourceMapString);
        } else if (msg.type === "error") {
            sourceMap = null;
            console.log("Could not calculate source map due to error:", msg.error);
        } else {
            console.log("Unknown message from source map generator: ", msg);
            throw new Error("Unknown message from sourceMapGenerator: " + msg.type);
        }
        
        sourceMapGenerator = undefined;
    });
    sourceMapGenerator.on("exit", function (code, signal) {
        sourceMapGenerator = undefined;
        //console.log("Source map generator process exited with code: " + code + " after receiving signal: " + signal);
        if (code !== 0) {
            // some error occured
            sendToParent({ type: "error", message: "Error instrumenting code: " + "SourceMap could not be generated" });
        }
    });

    sourceMapGenerator.send(code);
}

function startExecution(code, options) {
    runtime = fork("./runtime");

    runtime.on("message", function (message) {
        switch (message.type) {
        case "message array":
            message.stringifiedMessages.forEach(function (aMessageString) {
                var parsedMessage = JSON.parse(aMessageString);
                messagesToCorrect.push(parsedMessage);
            });
            break;
        default:
            // nothing to do here, just forward it
            sendToParent(message);
        }
    });
    runtime.on("exit", function (code, signal) {
        runtime = undefined;
        if (code === 0) {
            // seems to have exited normally
            // tell the client
            messagesToCorrect.push({type: "program finished"});
            // we do not exit since we first have to send all the messages
            // we simply wait to be killed by the server
            return;
        }
        
        if (code === SHARED_CONSTANTS.SEND_ERROR_CODE) {
            sendToParent({type: "error", message: "Execution process exited due to send error."});
            return;
        }
        
        console.log("Child process exited with code: " + code + " after receiving signal: " + signal);
        if (code !== 0) {
            // some error occured
            messagesToCorrect.push({
                type: "executionError",
                error: {message: "Process exited abnormally with code " + code + " after receiving signal: " + signal}
            });
        }
    });
    runtime.on("disconnect", function () {
        console.log("Execution process disconnected...");
        // ignore... we just send of our messages when they are ready and wait to be killed by the server
    });
    runtime.on("error", function (error) {
        console.log("Execution process encountered error: ", error);
        // TODO: Think about sending it to the client. Depends on what kinds of errors reach this method.
    });

    runtime.send({ type: "code", code: code, options: options });
}

process.on("message", function (msg) {
    if (msg.type === "code") {
        startExecution(msg.code, msg.options); // forks of a process to instrument and execute the code
        startSourceMapGenerator(msg.code); // forks of a process to calculate a source map from instrumented to original code
    }
});





// processing messages

function correctMessages() {
    function tryToCorrectAndOrExtractLocationOfMessageIfNecessary(message) {
        if ((message.loc !== undefined)
                && ((message.type === "uncaught exception") || (message.type === "caught exception") || (message.type === "console log") || (message.type === "console error"))) {
            
            if (!sourceMap) {
                // if we don't have a source map yet, we wan't correct a location
                // return false for now, to indicate the failure and try again later
                return false;
            }
            
            message.loc.start = sourceMap.originalPositionFor(message.loc.start);
            message.loc.end = sourceMap.originalPositionFor(message.loc.end);
            if ((typeof message.loc.start.line !== "number") || (typeof message.loc.end.line !== "number")) {
                delete message.loc;
            }
        }
        return true;
    }
    
    var successfullyCorrectedMessage = true;
    while (successfullyCorrectedMessage && (messagesToCorrect.length > 0)) {
        var nextMessageToCorrect = messagesToCorrect.shift();
        successfullyCorrectedMessage = tryToCorrectAndOrExtractLocationOfMessageIfNecessary(nextMessageToCorrect);
        if (successfullyCorrectedMessage) {
            correctedMessages.push(nextMessageToCorrect);
        } else {
            messagesToCorrect.unshift(nextMessageToCorrect);
        }
    }
}

function sendCorrectedMessage() {
    if (correctedMessages.length > 0) {
        try {
            sendToParent({ type: "message array", messages: correctedMessages});
        } catch (sendException) {
            console.log("Could not send messages to parent due to error: ", sendException);
            console.log("Stack: ", helper.extractInterestingValuesFromStack(sendException.stack));
            console.log("Buffered messages: ", correctedMessages);
        }
        correctedMessages = [];
    }
}

function processMessages() {
    correctMessages();
    sendCorrectedMessage();
}


// for testing purposes
// messagesToCorrect.push({type: "uncaught exception", exception: {message: "Bla"}, loc: {start: {line: 2, colum: 3}, end: {line: 3, colum: 1}}});

function messageProcessingLoop() {
    processMessages();
    setTimeout(messageProcessingLoop, WAIT_TIME_BETWEEN_MESSAGEPROCESSING);
}
process.nextTick(messageProcessingLoop);

process.on("uncaught exception", function (exception) {
    console.log("An uncaught exception occured in child process and was caught by process: ", exception);
    console.log("Stack: ", helper.extractInterestingValuesFromStack(exception.stack));
    console.log("Exiting process");
    process.exit(1);
});

process.on("exit", function () {
    //console.log("Execution manager exiting.");
    if (runtime) {
        runtime.kill("SIGKILL");
    }
    if (sourceMapGenerator) {
        sourceMapGenerator.kill("SIGKILL");
}   
});
process.on("disconnect", function () {
    //console.log("Execution manager was disconnected.");
    process.exit(0);
});