/*jslint vars: true, plusplus: true, devel: true, nomen: true, node: true, indent: 4, maxerr: 50 */
/*global define, $, document, window, brackets  */

/**
 * An object to hold Esprima results and allow querying them.
 */


"use strict";

// Helper functions

//returns -1 if first > second, 0 if first == second, 1 if second > first
function locationComparison(first, second) {
    if ((first.line === second.line) && (first.column === second.column)) {
        return 0;
    } //else 
    if (first.line > second.line) {
        return -1;
    } //else 
    if ((first.line === second.line) && (first.column > second.column)) {
        return -1;
    } //else
    return 1;
}


var OBJECT_AST_PROPERTIES = ["init", "expression", "left", "right", "callee", "object", "property", "test", "update", "argument", "value", "consequent", "alternate", "block", "finalizer"],
    ARRAY_AST_PROPERTIES = ["declarations", "arguments", "properties", "elements", "handlers"],
    OBJECT_OR_ARRAY_AST_PROPERTIES = ["body"];

/**
* add a parent property to each of the tree objects pointing to its parent
*/
function _setupTreeForTraversal(treeObject, parent) {
    treeObject.parent = parent;
    treeObject.filter = _setupTreeForTraversal.filter;
    treeObject._removeParentRelationShipForTree = _setupTreeForTraversal._removeParentRelationShipForTree;
    treeObject.isTraversable = true;
    
    OBJECT_OR_ARRAY_AST_PROPERTIES.forEach(function (propertyName) {
        if (treeObject[propertyName] instanceof Array) {
            treeObject[propertyName].forEach(function (childObject) {
                _setupTreeForTraversal(childObject, treeObject);
            });
        } else if (treeObject[propertyName] instanceof Object) {
            _setupTreeForTraversal(treeObject[propertyName], treeObject);
        }
    });
    
    ARRAY_AST_PROPERTIES.forEach(function (propertyName) {
        if (treeObject[propertyName] instanceof Array) {
            treeObject[propertyName].forEach(function (childObject) {
                _setupTreeForTraversal(childObject, treeObject);
            });
        }
    });
    
    OBJECT_AST_PROPERTIES.forEach(function (propertyName) {
        if (treeObject[propertyName] instanceof Object) {
            _setupTreeForTraversal(treeObject[propertyName], treeObject);
        }
    });
}
    
    /**
    * @return an array of all tree objects for which the given test function returns true
    */
_setupTreeForTraversal.filter = function (testFunction) {
    var that = this,
        matchingChildrenAndSelf = [];
    // check if this should be added
    if (testFunction(this)) {
        matchingChildrenAndSelf.push(this);
    }
    
    // recursive call to gather all the children
    OBJECT_OR_ARRAY_AST_PROPERTIES.forEach(function (propertyName) {
        var arrayOfMatchingChildrenArrays;
        if (that[propertyName] instanceof Array) {
            arrayOfMatchingChildrenArrays = that[propertyName].map(function (treeObject) {return treeObject.filter ? treeObject.filter(testFunction) : []; });
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat.apply(matchingChildrenAndSelf, arrayOfMatchingChildrenArrays);
        } else if (that[propertyName] instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(that[propertyName].filter ? that[propertyName].filter(testFunction) : []);
        }
    });
    
    ARRAY_AST_PROPERTIES.forEach(function (propertyName) {
        var arrayOfMatchingChildrenArrays;
        if (that[propertyName] instanceof Array) {
            arrayOfMatchingChildrenArrays = that[propertyName].map(function (treeObject) {return treeObject.filter ? treeObject.filter(testFunction) : []; });
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat.apply(matchingChildrenAndSelf, arrayOfMatchingChildrenArrays);
        }
    });
    
    OBJECT_AST_PROPERTIES.forEach(function (propertyName) {
        if (that[propertyName] instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(that[propertyName].filter ? that[propertyName].filter(testFunction) : []);
        }
    });
    
    return matchingChildrenAndSelf;
};



_setupTreeForTraversal._removeParentRelationShipForTree =  function () {
    this.filter(function (treeObject) {
        delete treeObject.parent;
    });
};

function EsprimaResult(newStructure) {
    this._structure = newStructure;
    _setupTreeForTraversal(this._structure);
}

EsprimaResult.EsprimaLocationFromCmLocation = function (cmLocation) {
    return { line: cmLocation.line + 1, column: cmLocation.ch };
};

EsprimaResult.CmLocationFromEsprimaLocation = function (esprimaLocation) {
    return { line: esprimaLocation.line - 1, ch: esprimaLocation.column };
};

// not needed afaik, just if subclassing another class and then it should be "= new SuperClass()" 
//EsprimaResult.prototype                 = new EsprimaResult(); 
//EsprimaResult.prototype.constructor     = EsprimaResult;

// EsprimaResult.prototype._structure      = {}; // useless if this._structure is always set in the constructor

EsprimaResult.prototype.getStructure = function () {
    return this._structure;
};

EsprimaResult.prototype.setStructure = function (newStructure) {
    this._structure = newStructure;
};

EsprimaResult.prototype.filter = function (testFunction) {
    return this._structure.filter(testFunction);
};

EsprimaResult.prototype.tokensInRange = function (start, end, typeFilter) {
    start   = EsprimaResult.EsprimaLocationFromCmLocation(start);
    end     = EsprimaResult.EsprimaLocationFromCmLocation(end);

    var result = [];
    var i;
    for (i = 0; i < this._structure.tokens.length; i++) {
        var token = this._structure.tokens[i];

        if (typeFilter !== null) {
            if (token.type !== typeFilter) {
                continue;
            }
        }

        if ((locationComparison(token.loc, start) <= 0) && (locationComparison(token.loc, end) >= 0)) {
            result.push(token);
        }
    }

    return result;
};

EsprimaResult.prototype.elementsInRangeByType = function (start, end, typeFilter) {
    start   = EsprimaResult.EsprimaLocationFromCmLocation(start);
    end     = EsprimaResult.EsprimaLocationFromCmLocation(end);

    var result = this.filter(function () {
        if (this.type !== typeFilter) {
            return false;
        }

        if ((locationComparison(this.loc.start, start) <= 0) && (locationComparison(this.loc.end, end) >= 0)) {
            return true;
        }

        return false;
    });

    return result;
};

EsprimaResult.prototype.blocklevelElementEndingInPosition = function (position) {
    var compareLineOnly = (position.column === undefined);

    //Esprima line indexes start at 1
    position = EsprimaResult.EsprimaLocationFromCmLocation(position);

    var result = this.filter(function () {
        //test if we have a block level element at all
        if (this.body === undefined) {
            return false;
        }

        //test if end is correct
        if (compareLineOnly) {
            return (this.loc.end.line === position.line);
        } //else
        return ((this.loc.end.line === position.line) && (this.loc.end.column === position.column));
    });

    return (result.length > 0) ? result[0] : undefined;
};

EsprimaResult.prototype.functionForPosition = function (position) {
    var compareLineOnly = (position.column === undefined);

    //Esprima line indexes start at 1
    position = EsprimaResult.EsprimaLocationFromCmLocation(position);

    var result = this.filter(function () {
        if (this.type !== "FunctionDeclaration") {
            return false;
        }

        var comparisonResult = (this.loc.start.line <= position.line) && (this.loc.end.line >= position.line);

        if (!compareLineOnly) {
            comparisonResult = comparisonResult && (this.loc.start.column <= position.column) && (this.loc.end.column >= position.column);
        }

        return comparisonResult;
    });

    return (result.length > 0) ? result[0] : undefined;
};


module.exports = EsprimaResult;
