/*jslint vars: true, plusplus: true, devel: true, nomen: true, regexp: true, indent: 4, maxerr: 50 */
/*global define, $, brackets, window, WebSocket */

define(function (require, exports, module) {
    "use strict";

    var CommandManager  = brackets.getModule("command/CommandManager"),
        AppInit         = brackets.getModule("utils/AppInit"),
        Menus           = brackets.getModule("command/Menus"),
        EditorManager   = brackets.getModule("editor/EditorManager"),

        wsUri = "ws://localhost:1234/",
        websocket = null,
        menu,
        previousMessageID = null;

    var
        START_LIVECODING = "Start Live Coding",
        START_LIVECODING_COMMAND_ID = "livecoding.startapp",
        STOP_RUNTIME = "Stop old runtime",
        STOP_RUNTIME_COMMAND_ID = "livecoding.stopruntime";

    function openSocket(uri, callOnOpen, callOnMsg) {
        var websocket = new WebSocket(uri);

        websocket.onclose = function (evt) { console.log("[brackets-livecoding]", "Closing websocket."); };
        websocket.onerror = function (evt) { console.log("[brackets-livecoding]", "Error: " + evt.data); };

        websocket.onopen = function (evt) { callOnOpen(); };
        websocket.onmessage = function (evt) { callOnMsg(evt.data); };

        return websocket;
    }

    // Prints the message received from the server
    function printMessage(msg) {
        var property, strings = ["[brackets-livecoding-server]"];

        function formatLocation(loc) {
            return "at " + loc.start.line + ":" + loc.start.column + " - " + loc.end.line + ":" + loc.end.column;
        }

        switch (msg.type) {
        case "variables init":
            strings.push("var ");
            msg.vars.forEach(function (init) {
                if (init.value === undefined) {
                    strings.push(init.name);
                } else {
                    strings.push(init.name + " =", init.value);
                }
            });
            strings.push(formatLocation(msg.loc));
            break;
        case "assignment":
        case "update":
            strings.push(msg.name + " =", msg.value, formatLocation(msg.loc));
            break;
        default:
            // Add all properties of the message object to the array
            for (property in msg) {
                if (msg.hasOwnProperty(property)) {
                    if (property === "loc") {
                        strings.push(formatLocation(msg.loc));
                    } else {
                        strings.push(property + ":", msg[property], ",");
                    }
                }
            }
        }

        // Print message to the console
        try {
            console.log.apply(console, strings);
        } catch (e) {
            console.error("[brackets-livecoding]", "Error printing message " + msg + ":", e.stack);
        }
    }

    // Sends message over the websocket. If a socket is not open, create one.
    function sendMessage(msg, id) {

        // Callback after socket opened succesfully
        function onOpen() {
            console.log("[brackets-livecoding]", "Opened websocket to " + wsUri + ".");
            websocket.send(JSON.stringify(msg));
            previousMessageID = id;
        }

        // Callback after receiving message back from server
        function onMsg(msg) {
            var i;
            try {
                i = msg.indexOf("#");
                msg = [msg.slice(0, i), msg.slice(i + 1)];

                if (msg[0] === id.toString()) {
                    msg = JSON.parse(msg[1]);

                    switch (msg.type) {
                    case "log":
                    case "error":
                        console[msg.type]("[brackets-livecoding-server]", msg.message);
                        break;
                    case "message array":
                        msg.messages.forEach(printMessage);
                        break;
                    default:
                        console.error("[brackets-livecoding]", "Unknown message type");
                    }
                }
            } catch (e) {
                console.error("[brackets-livecoding]", "Couldn't parse message:", e.stack);
            }
        }

        // Check if a websocket is open
        if (websocket && (websocket.readyState === WebSocket.OPEN)) {
            // Websocket open, send the data
            websocket.onmessage = function (evt) { onMsg(evt.data); };
            websocket.send(JSON.stringify(msg));
            previousMessageID = id;
        } else {
            // No websocket open, create new socket and send the data after it opens
            websocket = openSocket(wsUri, onOpen, onMsg);
        }
    }

    function stopOldRuntime() {
        if (previousMessageID !== null) {
            sendMessage({ type: "stopRuntime", runtimeID: previousMessageID });
        }

        previousMessageID = null;
    }

    function handleLiveCoding() {
        var doc = EditorManager.getActiveEditor().document.getText(false),
            timestamp = (new Date()).getTime(),
            codeMsg = {
                type:       "code",
                code:       doc,
                id:         timestamp,
                options:    { disposeAfterExit: true }
            };

        // Stop previous runtime
        stopOldRuntime();

        sendMessage(codeMsg, timestamp);
    }

    // We add the menu item only after Brackets has finished loading to make sure that the debug menu exists already
    AppInit.appReady(function () {
        // Register commands
        CommandManager.register(START_LIVECODING, START_LIVECODING_COMMAND_ID, handleLiveCoding);
        CommandManager.register(STOP_RUNTIME, STOP_RUNTIME_COMMAND_ID, stopOldRuntime);

        // Then create menu items bound to the command
        // The label of the menu items are the names we gave the commands (see above)
        menu = Menus.getMenu("debug-menu");
        menu.addMenuItem(START_LIVECODING_COMMAND_ID, "F11");
        menu.addMenuItem(STOP_RUNTIME_COMMAND_ID);
    });
});