/*jslint vars: true, plusplus: true, devel: true, nomen: true, todo: true, indent: 4, maxerr: 50, browser: true */
/*global define, CodeMirror, brackets, less, $, XMLHttpRequest, document, window, _ */

define(function (require, exports, module) {
    "use strict";

    require("underscore");
    var utils = require("utils");
    
        
    var $dataView,
        $backgroundView,
        $scrollView,
        dataSource,
        delegate,
        numberOfLines = 0,
        numberOfValuesAdded = 0,
        $lastAddedValueHolder,
        lastAddedValueLine,
        executionData,
        delayedUpdateTimer,
        selectedIterationsForRepeatableBlocks = {},
        elementsTheMouseIsIn = (function () {
            var elements = [];
            return {
                push: function (newValue) { elements.push(newValue); },
                forEach: function (someFunction) { elements.forEach(someFunction); },
                removeAll: function () { elements = []; },
                remove: function (objectToRemove) {
                    var indexOfObjectToRemove = elements.indexOf(objectToRemove);
                    if (indexOfObjectToRemove >= 0) {
                        elements.splice(indexOfObjectToRemove, 1);
                    }
                }
            };
        }()),
        expandedLines = [];
    
    var specialStrings = {
        functionPlaceholder : "__LIVECODING__FUNCTION",
        undefinedPlaceholder : "__LIVECODING__UNDEFINED",
        specialNumbersPlaceholder: {
            positiveInfinity: "__LIVECODING__POSITIVE_INFINITY",
            negativeInfinity: "__LIVECODING__NEGATIVE_INFINITY",
            notANumber: "__LIVECODING__NaN"
        },
        jQueryType : "__LIVECODING__JQUERY",
        htmlElementType: "__LIVECODING__HTML",
        dateType: "__LIVECODING__DATE"
    },
        MAXIMUM_NUMBER_OF_ELEMENTS_TO_DISPLAY_PER_LINE = 10,
        MAXIMUM_NUMBER_OF_CHARACTERS_TO_DISPLAY_PER_STRING = 50,
        INITIAL_SLOPE_FOR_ITERATION_SCRUBBER = 1 / 10;
    
    var DELAY_FOR_THROWING_AWAY_OLD_DATA = 300;
    
    exports.setPreviewLinesContainer = function setPreviewLinesContainer($newPreviewLinesContainer) {
        $newPreviewLinesContainer.empty();
        $scrollView = $newPreviewLinesContainer.scrollView;
        $backgroundView = $("<div class='lc-background-div'/>");
        $newPreviewLinesContainer.prepend($backgroundView);
        $dataView = $("<div class='liveData'/>");
        $newPreviewLinesContainer.append($dataView);
    };
    
    function _clearViewConfigurationData() {
        numberOfValuesAdded = 0;
        $lastAddedValueHolder = undefined;
        expandedLines = [];
    }
    
    function _clearPreview() {
        // the following is a fix to make sure the elements that are removed now are sent the correct mouseleave events
        elementsTheMouseIsIn.forEach(function ($anElementTheMouseIsIn) {
            $anElementTheMouseIsIn.mouseleave();
        });
        
        // clear preview
        $dataView.empty();
        $backgroundView.removeClass("lc-syntax-error-prevents-preview");
        $dataView.removeClass("lc-hasError");
        $backgroundView.empty();
        var i;
        for (i = 0; i < numberOfLines; i++) {
            $dataView.append("<pre class='lc-line-data'> </pre>"); // the space is important to force display of the pre element
        }
    }
    
    // creates a div, that has a fixed position in the assistant pane
    function _$getMessagesDiv() {
        var $messagesDiv =  $("<div class='messages'/>");
        var offsetOfLiveDataView = $dataView.offset(),
            offsetOfMessagesDiv = {};
        offsetOfMessagesDiv.top = offsetOfLiveDataView.top + 50;
        offsetOfMessagesDiv.left = offsetOfLiveDataView.left + 50;
        offsetOfMessagesDiv.right = 50;
        $messagesDiv.css(offsetOfMessagesDiv);
        return $messagesDiv;
    }
    
    function _clearErrors() {
        $(".liveData .messages").remove();
        $dataView.removeClass("lc-hasError");
    }
    
    function _displayGeneralError(error) {
        $dataView.addClass("lc-hasError");
        var $messagesDiv = _$getMessagesDiv();
        $messagesDiv.append("<span class='error'>" + error.message + "</span>");
        if (error.stack && error.stack !== specialStrings.undefinedPlaceholder) {
            $messagesDiv.append("<div class='stack'>" 
                                + error.stack.reduce(function (prev, curr) {return prev + "</br>" + curr; }, "")
                                + "</div>");
        }
        $dataView.prepend($messagesDiv);
    }
    
    function _setNoDataToDisplay() {
        var $messagesDiv = _$getMessagesDiv();
        if (dataSource.isExecutingCode()) {
            $messagesDiv.append("No data to display, yet.");
        } else {
            $messagesDiv.append("Execution finished. No data to display. Nothing seems to have been executed.");
        }
        $dataView.append($messagesDiv);
    }
    
    function _getSelectedIterationIndexForRepetableBlock(repeatableBlock) {
        var selectedIterationIndex = selectedIterationsForRepeatableBlocks[repeatableBlock.identifier],
            maximumIterationIndexForBlock = repeatableBlock.iterations.length - 1;
        
        if (selectedIterationIndex === undefined) {
            // the identifier that is used to retrieve the location is calculated from the location
            // now when the location of the function changes, the identifier changes.
            // so it might be, that we saved the iteration, but the function location was changed.
            // Check whether we have a selection index that belongs to a function with a similar location
            var currentLocation = repeatableBlock.location,
                currentStart = currentLocation.start,
                currentEnd = currentLocation.end,
                similarLocations = [
                    {start: currentStart, end: {line: currentEnd.line + 1, ch: currentEnd.ch}},
                    {start: currentStart, end: {line: currentEnd.line, ch: currentEnd.ch + 1}},
                    {start: currentStart, end: {line: currentEnd.line - 1, ch: currentEnd.ch}},
                    {start: currentStart, end: {line: currentEnd.line, ch: currentEnd.ch - 1}},
                    {start: {line: currentStart.line + 1, ch: currentStart.ch}, end: currentEnd},
                    {start: {line: currentStart.line, ch: currentStart.ch + 1}, end: currentEnd},
                    {start: {line: currentStart.line - 1, ch: currentStart.ch}, end: currentEnd},
                    {start: {line: currentStart.line, ch: currentStart.ch - 1}, end: currentEnd},
                    {start: {line: currentStart.line + 1, ch: currentStart.ch}, end: {line: currentEnd.line + 1, ch: currentEnd.ch}},
                    {start: {line: currentStart.line - 1, ch: currentStart.ch}, end: {line: currentEnd.line - 1, ch: currentEnd.ch}}
                ],
                i;
            for (i = 0; i < similarLocations.length; i++) {
                var identifierForLocation = utils.calculateIdentifierFromLocation(similarLocations[i]);
                identifierForLocation +=  repeatableBlock.hasOwnProperty("declarationIndex") ? ":" + repeatableBlock.declarationIndex : "";
                selectedIterationIndex = selectedIterationsForRepeatableBlocks[identifierForLocation];
                if (selectedIterationIndex) {
                    selectedIterationsForRepeatableBlocks[repeatableBlock.identifier] = selectedIterationIndex;
                    delete selectedIterationsForRepeatableBlocks[identifierForLocation]; // remove the old data
                    break;
                }
            }
        }
        
        selectedIterationIndex = selectedIterationIndex || 0; //make sure it's not undefined
        
        
        if (["while-loop", "for-loop"].indexOf(repeatableBlock.type) > -1) {
            // one more iteration is possible, since for these types of loops there still happens stuff "after" the last iteration
            maximumIterationIndexForBlock += 1;
        }
        if (selectedIterationIndex > maximumIterationIndexForBlock) {
            return maximumIterationIndexForBlock;
        }
        if (selectedIterationIndex < 0) {
            return 0;
        }
        return selectedIterationIndex;
    }
    
    function _incrementSelectedIterationForRepetableBlock(repeatableBlock) {
        var selectedIteration = _getSelectedIterationIndexForRepetableBlock(repeatableBlock);
        selectedIterationsForRepeatableBlocks[repeatableBlock.identifier] = selectedIteration + 1;
        _updateView();
    }
    
    function _decrementSelectedIterationForRepetableBlock(repeatableBlock) {
        var selectedIteration = _getSelectedIterationIndexForRepetableBlock(repeatableBlock);
        selectedIterationsForRepeatableBlocks[repeatableBlock.identifier] = selectedIteration - 1;
        _updateView();
    }
    
    function _updateSelectedIterationForRepetableBlock(newSelectedIteration, repeatableBlock) {
        selectedIterationsForRepeatableBlocks[repeatableBlock.identifier] = newSelectedIteration;
        _updateView();
    }
    
    // returns a function f(x) = slope * x
    function getLinearlyIncreasingFunctionWithSlope(slope) {
        return function (value) {
            return value * slope;
        };
    }
    
    // returns a function f(x) = e^(x*ln(value)/position)
    function getExponentiallyIncreasingFunctionForValueAtPosition(value, position) {
        if (value <= 1) { throw new Error("Value must be greater than 1."); }
        if (position <= 0) { throw new Error("Position must be greater than 0."); }
        return function (x) {
            return Math.pow(Math.E, x * Math.log(value) / position);
        };
    }
    
    function getLinearlyThanExponentiallyIncreasingFunctionWithInitialSlopeAndValueAtPosition(slope, value, position) {
        if (slope <= 0) { throw new Error("Slope must be greater than 0."); }
        
        // if the maxValue is rather small (so it can easily be reached using linear growth) use only a linear function
        if ((slope * position) / 2 > value) {
            return getLinearlyIncreasingFunctionWithSlope(slope);
        }
        
        
        // we want to combine a linear function and an exponential one
        // small values will be handled by the linear function, but as soon as the exponential one
        // grows faster it takes over, so we "attach" the exponential function to the linear one at the position 
        // where they have the same slope. 
        // We will add a constant value to the exponential function, to make it continue at the same value the 
        // linear function left off at.
        var linearFunction = getLinearlyIncreasingFunctionWithSlope(slope),
            exponentialFunction = getExponentiallyIncreasingFunctionForValueAtPosition(value, position);
        
        // First we need to calculate when these two functions will have the same slope
        // let f be the linear one and g the exponential one
        // f(x) = slope * x
        // f'(x) = slope
        // g(x) = e^(x * ln(value)/position)
        // g'(x) = ln(value)/position * e^(x * ln(value)/position)
        
        // f'(x) == g'(x)
        // <=> slope == ln(value)/position * e^(x * ln(value)/position)
        //
        //                            slope
        //           position * ln( --------- )
        //                          ln(value)
        // <=> x == ----------------------------
        //                  ln(value)
        //
        // <=> x == position * ln(position * slope/ ln(value)) / ln(value)
        var sameSlopePosition = position * Math.log(position * slope / Math.log(value)) / Math.log(value),
            offset = linearFunction(sameSlopePosition) - exponentialFunction(sameSlopePosition);
        
        
        return function (x) {
            if (x <= sameSlopePosition) {
                return linearFunction(x);
            } //else 
            return exponentialFunction(x) + offset;
        };
    }
     
    function _startScrubbingForRepeatableBlock(repeatableBlock, event) {
        var downX = event.pageX;
        var originalValue = _getSelectedIterationIndexForRepetableBlock(repeatableBlock),
            lastValue = originalValue;
        var maxValue = repeatableBlock.iterations.length - 1, // it needs to be at least 2 for the exponential function to work
            maxScrubPosition = Math.min(($(window).width() - downX - 1), downX), // the -1 in the first case is necessary to make sure that we reach our max value at the edge of the screen and aren't stopped 1 pixel before it.
            deltaCalculationFunction = getLinearlyThanExponentiallyIncreasingFunctionWithInitialSlopeAndValueAtPosition(INITIAL_SLOPE_FOR_ITERATION_SCRUBBER, maxValue, maxScrubPosition);
        console.log("maxValue: ", maxValue, " maxScrubPosition: ", maxScrubPosition);
        
        function movedDistanceSinceMousedown(event) {
            return event.pageX - downX;
        }
        
        
        function newValueForMovedDistance(movedDistance) {
            var sign = movedDistance < 0 ? -1 : 1;
            return originalValue + sign * Math.round(deltaCalculationFunction(Math.abs(movedDistance)));
        }
        
        function moveHandler(event) {
            var newValue = newValueForMovedDistance(movedDistanceSinceMousedown(event));
            
            if (newValue !== lastValue) {
                lastValue = newValue;
                _updateSelectedIterationForRepetableBlock(newValue, repeatableBlock);
            }
            
            return false;
        }
        
        function upHandler(event) {
            $(window.document).off("mousemove", moveHandler);
            $(window.document).off("mouseup", upHandler);
        }
        
        $(window.document).on("mousemove.iterationScrubbing", moveHandler);
        $(window.document).on("mouseup.iterationScrubbing", upHandler);
    }
    
    function _iterationSelectorUIForRepetableBlock(repeatableBlock) {
        var $leftArrow = $("<span class ='lc-iterationSelector'>&lt;</span>");
        var $rightArrow = $("<span class ='lc-iterationSelector'>&gt;</span>");
        var selectedIteration = _getSelectedIterationIndexForRepetableBlock(repeatableBlock);
        var $selectedIterationDisplay = $("<span class='lc-selectedIteration'>" + (selectedIteration + 1) + "</span>");
        var $iterationDisplay = $("<span class='lc-iterationDisplay'/>");
        var iterationCount = repeatableBlock.iterations.length;
        var $iterationCountDisplay = $("<span class='lc-iterationCount'>" + iterationCount + "</span>");
        var $iterationSelectorUI = $("<span class='lc-iterationSelectionUI'/>");
        $leftArrow.on("click", function () {
            _decrementSelectedIterationForRepetableBlock(repeatableBlock);
        });
        $rightArrow.on("click", function () {
            _incrementSelectedIterationForRepetableBlock(repeatableBlock);
        });
        
        $iterationDisplay.append($selectedIterationDisplay);
        $iterationDisplay.append($iterationCountDisplay);
        $iterationDisplay.on("mousedown.iterationScrubbing", function (event) {
            if (event.which === 1) {
                _startScrubbingForRepeatableBlock(repeatableBlock, event);
            }
            return false;
        });
        
        $iterationSelectorUI.append($leftArrow);
        $iterationSelectorUI.append($iterationDisplay);
        $iterationSelectorUI.append($rightArrow);
        
        if (repeatableBlock.hasUncaughtException) {
            $iterationSelectorUI.addClass("hasUncaughtException");
            var iterationIndexesWithUncaughtExceptions = repeatableBlock.iterations.forEach(function (iterationBlock, index) {
                if (iterationBlock.hasUncaughtException) {
                    if (index === selectedIteration) {
                        $selectedIterationDisplay.addClass("hasUncaughtException");
                    } else if (index < selectedIteration) {
                        $leftArrow.addClass("hasUncaughtException");
                    } else {
                        $rightArrow.addClass("hasUncaughtException");
                    }
                }
            });
        }
        
        return $iterationSelectorUI;
    }
    
    function _displayElementForValueData(valueData, showCompleteValues, expandButtonCallback) {
        var indexOfElement = 0,
            $elementForValueData,
            somethingWasHidden = false;
        
        function abbreviatedStringHTMLForStringValue(theString) {
            var shortenedString;
            shortenedString = theString.substr(0, MAXIMUM_NUMBER_OF_CHARACTERS_TO_DISPLAY_PER_STRING);
            if (showCompleteValues) {
                shortenedString = theString;
            }
            if (shortenedString !== theString) {
                somethingWasHidden = true;
            }
            return _.escape(shortenedString)
                    + (somethingWasHidden ? "<span class='expandButton'>...</span>" : "");
        }
        
        function _displayHTMLForValue(value) {
            indexOfElement++;
            var shortenedString, referenceString;
            var stringRepresentation, property, isFirstProperty = true;
            
            // first check for some of the special placeholder strings or null
            switch (value) {
            case specialStrings.undefinedPlaceholder:
                return "<span class='undefined-value'>undefined</span>";
            case specialStrings.functionPlaceholder:
                return "<span class='function-value'>function</span>";
            case specialStrings.specialNumbersPlaceholder.positiveInfinity:
                return "<span class='cm-number'>Infinity</span>";
            case specialStrings.specialNumbersPlaceholder.negativeInfinity:
                return "<span class='cm-number'>-Infinity</span>";
            case specialStrings.specialNumbersPlaceholder.notANumber:
                return "<span class='cm-number'>NaN</span>";
            case null:
                return "<span class='null-value'>null</span>";
            }
            if ((typeof value === "string") || value instanceof String) {
                
                return "<span class='cm-string'>\""
                        + abbreviatedStringHTMLForStringValue(value)
                        + "\"</span>";
            } // else
            if (typeof value === "number") {
                return "<span class='cm-number'>" + value + "</span>";
            } // else
            if (typeof value === "boolean") {
                return "<span class='" + value + "-value'>" + value + "</span>";
            } // else
            if ($.isArray(value)) {
                var arrayValuesHTMLString = value.reduce(function (previousValue, currentValue) {
                    if (!showCompleteValues && (indexOfElement > MAXIMUM_NUMBER_OF_ELEMENTS_TO_DISPLAY_PER_LINE)) {
                        somethingWasHidden = true;
                        return previousValue;
                    }
                    return previousValue + (previousValue  === "[" ? "" : ", ") + _displayHTMLForValue(currentValue);
                }, "[");
                stringRepresentation = "<span class='array-value'>"
                    + arrayValuesHTMLString
                    + (somethingWasHidden ? ", <span class='expandButton'>...</span>]" : "]")
                    + "</span>";
                return stringRepresentation;
            } // else
            if (!(typeof value === "object")) {
                throw new TypeError("Value of type " + typeof value + " can not be handled.");
            }
            
            // there are a few special objects, e.g. circular references
            if (value.hasOwnProperty("$ref") && Object.keys(value).length === 1) {
                // it's a reference object pointing to another location in its parent
                // we simply display the reference string, but give it a special highlight
                referenceString = value.$ref;
                // the reference string starts with a $ to reference the root and the a path to the object
                // we replace the "$" by "root" to make that more explicit
                referenceString = "root" + referenceString.substr(1);
                return "<span class='circular-reference'>circular(" + referenceString + ")</span>";
            }
            
            // or objects of a specific type like jquery or HTML
            if (value && (value.type === specialStrings.jQueryType)) {
                stringRepresentation = "<span class='jQuery-value'>";
                stringRepresentation += value.value.reduce(function (previousValue, currentValue) {
                    if (!showCompleteValues && (indexOfElement > MAXIMUM_NUMBER_OF_ELEMENTS_TO_DISPLAY_PER_LINE)) {
                        somethingWasHidden = true;
                        return previousValue;
                    }
                    return previousValue + (previousValue  === "[" ? "" : ", ") + abbreviatedStringHTMLForStringValue(currentValue);
                }, "[");
                stringRepresentation += "]" + "</span>";
                return stringRepresentation;
            }// else
            if (value && (value.type === specialStrings.htmlElementType)) {
                return "<span class='html-value'>"
                        + abbreviatedStringHTMLForStringValue(value.value)
                        + "</span>";
            } // else
            if (value && (value.type === specialStrings.dateType)) {
                return "<span class='date-value'>"
                        + abbreviatedStringHTMLForStringValue(value.value)
                        + "</span>";
            } // else
            
            stringRepresentation = "<span class='object'>{";
            for (property in value) {
                if (value.hasOwnProperty(property)) {
                    if (!showCompleteValues && (indexOfElement > MAXIMUM_NUMBER_OF_ELEMENTS_TO_DISPLAY_PER_LINE)) {
                        somethingWasHidden = true;
                        // do not modify the HTML-String anymore, ignore all other values
                    } else {
                        if (isFirstProperty) {
                            isFirstProperty = false;
                        } else {
                            stringRepresentation += ", ";
                        }
                        stringRepresentation += property + ": " + _displayHTMLForValue(value[property]);
                    }
                }
            }
            if (somethingWasHidden) { stringRepresentation += ", <span class='expandButton'>...</span>"; }
            stringRepresentation += "}</span>";
            return stringRepresentation;
        }
        
        
        if (valueData.type === "error" || valueData.type === "caught error") {
            $elementForValueData = $("<span class='" + valueData.type + "'>" + abbreviatedStringHTMLForStringValue(valueData.message) + "</span>");
        } else if (valueData.type === "catch") {
            $elementForValueData = $("<span class='" + valueData.type + "'><span class='name'>caught:</span> " + abbreviatedStringHTMLForStringValue(valueData.parameter.message) + "</span>");
        } else if (!valueData.isRepeatableBlock) {
            var booleanAttributeString = "";
            if (valueData.type.indexOf("test")) {
                booleanAttributeString = "boolean='" + valueData.result + "'";
            }
            var HTMLStringForDisplayElement = "<span class='" + valueData.type + "' " + booleanAttributeString + ">" + _displayHTMLForValue(valueData.value) + "</span>";
            $elementForValueData = $(HTMLStringForDisplayElement);
        } else if (valueData.type) {
            // it's a repeatable block. Display UI to enable the selection of an iteration.
            return _iterationSelectorUIForRepetableBlock(valueData);
        }
        if ($elementForValueData) {
            if (somethingWasHidden) {
                var $expandButton = $elementForValueData.find(".expandButton");
                $expandButton.click(expandButtonCallback);
            }
            return $elementForValueData;
        }
    }
    
    function _addDisplayForValueToLineElement(valueData, $lineDataView) {
        var $valueHolder = $("<span title='" + valueData.name + "'>"),
            showCompleteValues = false;
        if (expandedLines.indexOf($lineDataView.lineIndex) > -1) {
            showCompleteValues = true;
        }
        $valueHolder.append(_displayElementForValueData(valueData, showCompleteValues, function (clickEvent) {
            console.log("expand line " + $lineDataView.lineIndex);
            expandedLines.push($lineDataView.lineIndex);
            _updateView();
        }));
        $valueHolder.mouseenter(function () {
            delegate.mouseEnteredValueDisplayForValueData(valueData);
            elementsTheMouseIsIn.push($valueHolder);
        });
        $valueHolder.mouseleave(function () {
            delegate.mouseLeftValueDisplayForValueData(valueData);
            elementsTheMouseIsIn.remove($valueHolder);
        });
        $valueHolder.addClass("valueHolder");
        
        $lineDataView.append($valueHolder);
    }
    
    function _createValueAdderForLineElement($lineElement) {
        return function (valueData) {
            _addDisplayForValueToLineElement(valueData, $lineElement);
        };
    }
    
    function objectIsRepeatableBlock(candidate) {
        return candidate.isRepeatableBlock === true;
    }
    
    function _findRepeatableBlockForLineInExecutionData(lineIndex, executionDataToCheck) {
        
        while (lineIndex >= 0) {
            if (executionDataToCheck.body[lineIndex] && executionDataToCheck.body[lineIndex].length > 0) {
                // find the first repeatable block in this line
                var repeatableBlockInCurrentLine = executionDataToCheck.body[lineIndex].filter(objectIsRepeatableBlock)[0];
                if (repeatableBlockInCurrentLine !== undefined) {
                    return repeatableBlockInCurrentLine;
                }// else
                return null;
            }
            lineIndex--;
        }
        return null;
    }
    
    function _getSelectedIterationBlockForRepeatableBlock(repeatableBlock) {
        var selectedIteration = _getSelectedIterationIndexForRepetableBlock(repeatableBlock);
        if (selectedIteration === repeatableBlock.iterations.length) {
            // that's a special case and should only happen for for-loops and while-loops
            if (["while-loop", "for-loop"].indexOf(repeatableBlock.type) < 0) {
                if (repeatableBlock.iterations.length > 0) {
                    // that's not supposed to happen
                    throw new TypeError("Iteration Index " + selectedIteration + " for repetable block of type '" + repeatableBlock.type + " is equal to the number of iterations of the block.");
                } else {
                    // we reached this point by accident, there are simply no iterations
                    return undefined;
                }
            } // else
            // return a placeholder iteration block, that does not contain any execution data (since it didn't happen) 
            // but contains a reference to this repetable block and thus enables us to search backwards in the other iteration blocks 
            // for the update and condition values that should be displayed for this iteration
            return {body: [], parent: repeatableBlock, type: repeatableBlock.type + " body"};
        }
        return repeatableBlock.iterations[selectedIteration];
    }
    
    function _findInitValuesForLineInExecutionData(lineIndex, executionDataToCheck) {
        if (!executionDataToCheck) {
            return [];
        }
        
        var initDataForLine, parentRepeatableBlock, previousIteration, selectedIterationIndex;
        if (executionDataToCheck.type === "for-loop body") {
            parentRepeatableBlock = executionDataToCheck.parent;
            selectedIterationIndex = _getSelectedIterationIndexForRepetableBlock(parentRepeatableBlock);
            if (selectedIterationIndex === 0) {
                // we have to use the init- and test-property of the for-loop
                initDataForLine = parentRepeatableBlock.init[lineIndex];
                if (initDataForLine) {
                    initDataForLine = initDataForLine.slice(0); // copy the array
                } else {
                    initDataForLine = [];
                }
                if (parentRepeatableBlock.test && (parentRepeatableBlock.test.location.start.line === lineIndex)) {
                    initDataForLine.push(parentRepeatableBlock.test);
                }
            } else {
                // use the update- and test-property of the previous execution (special case)
                previousIteration = parentRepeatableBlock.iterations[_getSelectedIterationIndexForRepetableBlock(parentRepeatableBlock) - 1];
                initDataForLine = previousIteration.update && previousIteration.update.body[lineIndex];
                if (initDataForLine) {
                    initDataForLine = initDataForLine.slice(0); // copy the array
                } else {
                    initDataForLine = [];
                }
                if (previousIteration.test && (previousIteration.test.location.start.line === lineIndex)) {
                    initDataForLine.push(previousIteration.test);
                }
            }
        } else if (executionDataToCheck.type === "while-loop body") {
            parentRepeatableBlock = executionDataToCheck.parent;
            selectedIterationIndex = _getSelectedIterationIndexForRepetableBlock(parentRepeatableBlock);
            if (selectedIterationIndex === 0) {
                // we have to use the test-property of the while-loop itself
                if (parentRepeatableBlock.test && (parentRepeatableBlock.test.location.start.line === lineIndex)) {
                    initDataForLine = [parentRepeatableBlock.test];
                }
            } else {
                // use the test property of the previous execution
                previousIteration = parentRepeatableBlock.iterations[_getSelectedIterationIndexForRepetableBlock(parentRepeatableBlock) - 1];
                if (previousIteration.test  && (previousIteration.test.location.start.line === lineIndex)) {
                    initDataForLine = [previousIteration.test];
                }
            }
        } else if ((executionDataToCheck.type === "for-in-loop body") && (executionDataToCheck.loopVarLocation.start.line === lineIndex)) {
            // for a for-in-loop we only have one loop variable for which we want to show the value
            initDataForLine = [{
                name: executionDataToCheck.loopVarName,
                value: executionDataToCheck.loopVarValue,
                location: executionDataToCheck.loopVarLocation,
                type: "for-in-loop variable"
            }];
        } else if (executionDataToCheck.type === "function body") {
            initDataForLine = executionDataToCheck.argumentValues.map(function (aValue, index) {
                return {
                    value: aValue,
                    name: executionDataToCheck.argumentNames[index] || "arguments[" + index + "]",
                    location: executionDataToCheck.argumentLocations && executionDataToCheck.argumentLocations[index],
                    type: "function parameter"
                };
            });
            var unnamedArgumentsLine = Math.max(executionDataToCheck.bodyLocation.start.line - 1, executionDataToCheck.location.start.line);
            initDataForLine = initDataForLine.filter(function (parameterValue) {
                if (parameterValue.location) {
                    // that next bit is a bit hacky. Well, a lot, actually.
                    // We adjust the unnamedArgumentsLine upwards if we find a parameters with a location, 
                    // to ensure unnamed parameters (which don't have a location) appear after named parameters (which do have a location)
                    // This should work since the named parameters appear in the array first
                    unnamedArgumentsLine = Math.max(parameterValue.location.start.line, unnamedArgumentsLine);
                    return parameterValue.location.start.line === lineIndex;
                } // else
                return unnamedArgumentsLine === lineIndex;
            });
        }
        
        if ((initDataForLine === undefined) || (initDataForLine.length <= 0)) {
            return [];
        } // else
        return initDataForLine;
    }
    
    function _findPreviewValuesForLineInExecutionData(lineIndex, executionDataToCheck) {
        if (!executionDataToCheck) {
            return [];
        }
        
        var previewValues;
        var executionDataForLine = executionDataToCheck.body[lineIndex];
        var repeatableBlock;
        var valuesForThisLine;
        var initValuesForthisLine;
        
        if ((executionDataForLine === undefined) || (executionDataForLine.length === 0)) {
            // line without preview values
            // this might be due to this line being part of a loop or function
            // check whether we find a repeatable block that covers this line
            repeatableBlock = _findRepeatableBlockForLineInExecutionData(lineIndex, executionDataToCheck);
            if ((repeatableBlock !== null) && (repeatableBlock.type)) {
                valuesForThisLine = _findPreviewValuesForLineInExecutionData(lineIndex, _getSelectedIterationBlockForRepeatableBlock(repeatableBlock));
                initValuesForthisLine = _findInitValuesForLineInExecutionData(lineIndex, _getSelectedIterationBlockForRepeatableBlock(repeatableBlock));
                return initValuesForthisLine.concat(valuesForThisLine);
            } // else
            return [];
        } // else
        if (executionDataForLine.length > 0) {
            if (!executionDataForLine.some(objectIsRepeatableBlock)) {
                // none of the values on this line is a repetable block
                return executionDataForLine;
            } //else
            // find the first repetable block
            repeatableBlock = executionDataForLine.filter(objectIsRepeatableBlock)[0];
            
            // Go in there and find the values for this line
            valuesForThisLine = _findPreviewValuesForLineInExecutionData(lineIndex, _getSelectedIterationBlockForRepeatableBlock(repeatableBlock));
            initValuesForthisLine = _findInitValuesForLineInExecutionData(lineIndex, _getSelectedIterationBlockForRepeatableBlock(repeatableBlock));
            
            return executionDataForLine.concat(initValuesForthisLine, valuesForThisLine);
        } //  else
        return [];
    }
    
    function _displayExecutionData() {
        var lineIndex, previewValuesForCurrentLine;
        for (lineIndex = 0; lineIndex < numberOfLines; lineIndex += 1) {
            previewValuesForCurrentLine = _findPreviewValuesForLineInExecutionData(lineIndex, executionData);
            var $lineDataView = $("<div/>");
            $lineDataView.lineIndex = lineIndex;
            previewValuesForCurrentLine.forEach(_createValueAdderForLineElement($lineDataView));
            $lineDataView.appendTo($dataView.find("pre.lc-line-data")[lineIndex]);
        }
    }
    
    exports.setNumberOfLines = function _setNumberOfLines(newNumberOfLines) {
        numberOfLines = newNumberOfLines;
    };
    
    exports.highlightLines = function (linesToHighlight) {
        $(".lc-dataView-highlightedLines").removeClass("lc-dataView-highlightedLines");
        linesToHighlight.forEach(function (lineIndex) {
            $dataView.find(".lc-line-data:eq(" + lineIndex + ")").addClass("lc-dataView-highlightedLines");
        });
    };
    
    
    /* needed for parameter input for functions, currently not implemented
    
    function _valueForString(stringValue) {
        stringValue = stringValue.trim();
        if (stringValue.length <= 0) {
            return undefined;
        } // else
        if ("undefined" === stringValue) {
            return undefined;
        }
        return JSON.parse(stringValue);
    }
    
    exports.provideInputForParameterOfFunctionInLine = function provideInputForParameterOfFunctionInLine(parameterObject, context) {
        var $parameterInputSpan = $("<span class='parameterInput'>" + parameterObject.name + ": </span>");
        var $parameterInputValue = $("<span class='parameterInputValue' contentEditable='true'/>");
        $parameterInputValue.appendTo($parameterInputSpan);
        
        $parameterInputValue.setInputValue = function (value) {
            $parameterInputValue.removeClass("lc-non-parseable");
            if (value === undefined) {
                $parameterInputValue.addClass("lc-undefined");
                $parameterInputValue.text("undefined");
            } else {
                $parameterInputValue.removeClass("lc-undefined");
                $parameterInputValue.text(JSON.stringify(value));
            }
        };
        $parameterInputValue.setInputValue(parameterObject.value);
        
        
        var isAHackyBlur = false;
        $parameterInputValue.focusin(function () {
            var placeHolderTextForUndefined = "";
            if (($parameterInputValue.text() !== placeHolderTextForUndefined)
                    && $parameterInputValue.hasClass("lc-undefined")
                    && !$parameterInputValue.hasClass("lc-non-parseable")) {
                $parameterInputValue.text(placeHolderTextForUndefined);
                isAHackyBlur = true;
                $parameterInputValue.blur();
                isAHackyBlur = false;
                $parameterInputValue.focus(); // need to refocus
            }
        });
        
        $parameterInputValue.on("input", function () {
            try {
                var value = _valueForString($parameterInputValue.text());
                
                $parameterInputValue.removeClass("lc-non-parseable");
            } catch (exception) {
                $parameterInputValue.addClass("lc-non-parseable");
            }
        });
        
        $parameterInputValue.focusout(function (event) {
            if (isAHackyBlur) {
                // ignore it
                return;
            }
            var value;
            try {
                value = _valueForString($parameterInputValue.text());
                
                parameterObject.value = value;
                $parameterInputValue.setInputValue(value);
            } catch (exception) {
                $parameterInputValue.addClass("lc-non-parseable");
            }
        });
        
        $parameterInputSpan.appendTo($dataView.find("pre.lc-line-data")[parameterObject.line]);
        
//        $parameterInputSpan.mouseenter(function (event) {
//            //                              v- entered
//            mouseEnteredLeftHandler(event, true, context);
//        });
//        $parameterInputSpan.mouseleave(function (event) {
//            //                              v- left
//            mouseEnteredLeftHandler(event, false, context);
//        });
    };
    */
    
    exports.setDataSource = function (newDataSource) {
        dataSource = newDataSource;
    };
    
    exports.setDelegate = function (newDelegate) {
        delegate = newDelegate;
    };
    
    function _updateView() {
        _clearPreview();
        _displayExecutionData();
    }
    
    exports.reloadData = function (throwAwayOldDataDirectly) {
        // it might be that we delayed an update ealier. 
        // We don't need the delayed update anymore if we are updating now, so we cancel it
        if (delayedUpdateTimer) {
            window.clearTimeout(delayedUpdateTimer);
            delayedUpdateTimer = null;
        }
        
        
        if (dataSource.hasError()) {
            numberOfLines = dataSource.getNumberOfLinesOfExecutedCode();
            _clearErrors();
            console.log("reloading data: showing error message");
            _displayGeneralError(dataSource.getError());
            
        } else if (!dataSource.hasExecutionData()) {
            if (dataSource.isExecutingCode() && !throwAwayOldDataDirectly) {
                // if we are currently executing code it might just be, that we didn't wait long enough
                // for data from the backend, so we wait a little longer and keep the old data a little
                // longer to avoid flickering of the preview
                // This way we update when we have new data immediately, but don't throw away old data prematurely
                delayedUpdateTimer = window.setTimeout(function () {
                    delayedUpdateTimer = null;
                    // after a delay, we simply reload again, but definitely show the current state,
                    // even if there is no data
                    exports.reloadData(true);
                }, DELAY_FOR_THROWING_AWAY_OLD_DATA);
            } else {
                numberOfLines = dataSource.getNumberOfLinesOfExecutedCode();
                _clearPreview();
                console.log("reloading data: no execution data");
                _setNoDataToDisplay();
            }
        } else {
            numberOfLines = dataSource.getNumberOfLinesOfExecutedCode();
            _clearViewConfigurationData();
            console.log("reloading data: " + new Date());
            executionData = dataSource.getExecutionData();
            
            _updateView();
        }
        $(".lc-executing-indicator").remove();
        if (dataSource.isExecutingCode()) {
            $("<span class='lc-executing-indicator'>&nbsp;</span>").prependTo($dataView.find("pre.lc-line-data")[dataSource.getLastExecutedLineIndex()]);
        }
        
        // update toolbar button display
        $(".toolbar #liveCodingToolbarButton").removeClass("executing").removeClass("error").removeClass("uncaught-exception");
        $(".toolbar #liveCodingToolbarButton").attr("title", "Live execution completed");
        if (dataSource.hasError()) {
            $(".toolbar #liveCodingToolbarButton").addClass("error");
            $(".toolbar #liveCodingToolbarButton").attr("title", "Could not execute code.");
        } else if (dataSource.hasExecutionData() && executionData.hasUncaughtException) {
            $(".toolbar #liveCodingToolbarButton").addClass("uncaught-exception");
            $(".toolbar #liveCodingToolbarButton").attr("title", "Encountered uncaught exception. Click to reveal it.");
        } else if (dataSource.isExecutingCode()) {
            $(".toolbar #liveCodingToolbarButton").addClass("executing");
            $(".toolbar #liveCodingToolbarButton").attr("title", "Executing code...");
        }
        
        $scrollView.scrollToFitSynchronizedScrollView();
        console.log("reloaded data: " + new Date());
    };
});