/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, maxerr: 50 */
/*global define, CodeMirror, brackets, less, $, XMLHttpRequest, document, window */

define(function (require, exports, module) {
    "use strict";
    
    var Evaluator       = require('LiveCodingEvaluator'),
        esprima         = require('esprima'),
        StatusBar       = brackets.getModule("widgets/StatusBar"),
        EsprimaResult   = require('EsprimaResult'),
        utils           = require('utils');
    
    var liveDataView,
        previewPane,
        documentToPreview,
        codeToPreview,
        functionsWithParameters = {},
        linesWithGeneratedFunctions = [];
    
    var currentRepeatableBlockOfCode = {type: "program"},
        encounteredRepetableBlocksOfCode = {},
        executionData = {topMostExecutionContext: currentRepeatableBlockOfCode,
                         numberOfExecutedStatements: 0,
                        functions: {}},
        lastExecutedLineIndex = 0,
        isExecuting = false,
        currentError;
    
    function _smallerThan(threshold) {
        return function (value) {
            return value < threshold;
        };
    }
    
    function setExecuting(newValueForIsExecuting) {
        if (isExecuting === newValueForIsExecuting) {
            return;
        }
        console.log("executing: " + newValueForIsExecuting);
        isExecuting = newValueForIsExecuting;
        if (isExecuting) {
            $("#lc-executing-status-indicator").show();
        } else {
            $("#lc-executing-status-indicator").hide();
        }
    }
    
    function selectionChanged(event) {
        var selection = event.currentTarget.getSelection();
        var i;
        var endOfSelectionLine = selection.end.line;
        if ((selection.end !== selection.start) && selection.end.ch === 0) {
            // the end of the selection is at the beginning of a non-selected line
            endOfSelectionLine -= 1;
        }
        var selectedLines = [];
        for (i = selection.start.line; i <= endOfSelectionLine; i += 1) {
            selectedLines.push(i);
        }
        liveDataView.highlightLines(selectedLines);
    }
    
    function markRepeatableBlockAsHavingUncaughtException(aRepeatableBlock) {
        aRepeatableBlock.hasUncaughtException = true;
        if (aRepeatableBlock.parent) {
            if (aRepeatableBlock.type === "function body") {
                markRepeatableBlockAsHavingUncaughtException(aRepeatableBlock.functionDeclaration);
            } else {
                markRepeatableBlockAsHavingUncaughtException(aRepeatableBlock.parent);
            }
        }
    }
    
    function newBitOfCodeWasExecuted(message) {
        if (message.type === "Server Connection") {
            if (message.reason === "closed") {
                currentError = {type: "NoServerConnection",
                                message: "No connection to server."};
            } else if (message.reason === "error") {
                currentError = {type: "ServerConnectionError",
                                message: message.message};
            }
            console.error("Server Error: ", currentError);
            setExecuting(false);
            return;
        } // else
        if (message.type === "executionError") {
            currentError = {type: "executionError",
                            message: message.error.message};
            setExecuting(false);
            return;
        }
        if ((message.type === "uncaught exception") && (!message.loc)) {
            currentError = {type: "uncaught exception",
                            message: message.exception.message,
                            stack: message.stack};
            setExecuting(false);
            return;
        }
        if (message.type === "program finished") {
            console.log("program finished");
            setExecuting(false);
            return;
        }
        
        currentError = undefined;
        
        if (message.location === undefined) {
            console.log(message);
            throw new TypeError("Message does not have a location: " + message.type);
        }
        
        // setup if necessary
        if (currentRepeatableBlockOfCode.body === undefined) {
            currentRepeatableBlockOfCode.body = [];
        }
        var lineContainer = currentRepeatableBlockOfCode.body;
        if (currentRepeatableBlockOfCode.type === "for-loop") {
            lineContainer = currentRepeatableBlockOfCode.init;
        }
        if (lineContainer[message.location.start.line] === undefined) {
            lineContainer[message.location.start.line] = [];
        }
        
        
        var arrayToPushTo = lineContainer[message.location.start.line];
        
        var newRepeatableBlock, error;
        switch (message.type) {
        case "for-loop init":
            newRepeatableBlock = {type: "for-loop"};
            newRepeatableBlock.isRepeatableBlock = true;
            newRepeatableBlock.location = message.location;
            newRepeatableBlock.iterations = [];
            newRepeatableBlock.init = [];
            newRepeatableBlock.identifier = utils.calculateIdentifierFromLocation(message.location);
            arrayToPushTo.push(newRepeatableBlock);
            newRepeatableBlock.parent = currentRepeatableBlockOfCode;
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
        case "for-in-loop init":
            newRepeatableBlock = {type: "for-in-loop"};
            newRepeatableBlock.isRepeatableBlock = true;
            newRepeatableBlock.location = message.location;
            newRepeatableBlock.iterations = [];
            newRepeatableBlock.init = [];
            newRepeatableBlock.identifier = utils.calculateIdentifierFromLocation(message.location);
            arrayToPushTo.push(newRepeatableBlock);
            newRepeatableBlock.parent = currentRepeatableBlockOfCode;
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
        case "while-loop init":
            newRepeatableBlock = {type: "while-loop"};
            newRepeatableBlock.isRepeatableBlock = true;
            newRepeatableBlock.location = message.location;
            newRepeatableBlock.iterations = [];
            newRepeatableBlock.identifier = utils.calculateIdentifierFromLocation(message.location);
            arrayToPushTo.push(newRepeatableBlock);
            newRepeatableBlock.parent = currentRepeatableBlockOfCode;
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
                
        case "for-loop enter":
            newRepeatableBlock = {type: "for-loop body"};
            newRepeatableBlock.body = [];
            var forLoopBlock;
            if (currentRepeatableBlockOfCode.type === "for-loop") {
                forLoopBlock = currentRepeatableBlockOfCode;
            } else if (currentRepeatableBlockOfCode.type === "for-loop body") {
                forLoopBlock = currentRepeatableBlockOfCode.parent;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            newRepeatableBlock.parent = forLoopBlock;
            forLoopBlock.iterations.push(newRepeatableBlock);
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
        case "for-in-loop enter":
            newRepeatableBlock = {type: "for-in-loop body"};
            newRepeatableBlock.body = [];
            newRepeatableBlock.loopVarName = message.loopVar;
            newRepeatableBlock.loopVarValue = message.value;
            newRepeatableBlock.loopVarLocation = message.loopVarLocation;
                
            var forInLoopBlock;
            if (currentRepeatableBlockOfCode.type === "for-in-loop") {
                forInLoopBlock = currentRepeatableBlockOfCode;
            } else if (currentRepeatableBlockOfCode.type === "for-in-loop body") {
                forInLoopBlock = currentRepeatableBlockOfCode.parent;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            newRepeatableBlock.parent = forInLoopBlock;
            forInLoopBlock.iterations.push(newRepeatableBlock);
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
        case "while-loop enter":
            newRepeatableBlock = {type: "while-loop body"};
            newRepeatableBlock.body = [];
            var whileLoopBlock;
            if (currentRepeatableBlockOfCode.type === "while-loop") {
                whileLoopBlock = currentRepeatableBlockOfCode;
            } else if (currentRepeatableBlockOfCode.type === "while-loop body") {
                whileLoopBlock = currentRepeatableBlockOfCode.parent;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            newRepeatableBlock.parent = whileLoopBlock;
            whileLoopBlock.iterations.push(newRepeatableBlock);
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
                
        case "for-loop update enter":
            newRepeatableBlock = {type: "for-loop update"};
            newRepeatableBlock.body = [];
            var forLoopBodyBlock;
            if (currentRepeatableBlockOfCode.type === "for-loop body") {
                forLoopBodyBlock = currentRepeatableBlockOfCode;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            newRepeatableBlock.parent = forLoopBodyBlock;
            forLoopBodyBlock.update = newRepeatableBlock;
            currentRepeatableBlockOfCode = newRepeatableBlock;
            break;
        case "for-loop update exit":
            var forLoopUpdateToExit;
            if (currentRepeatableBlockOfCode.type === "for-loop update") {
                forLoopUpdateToExit = currentRepeatableBlockOfCode;
            } else {
                throw {name: "Unexpected execution block type",
                       infringingType: newRepeatableBlock.type};
            }
            currentRepeatableBlockOfCode = forLoopUpdateToExit.parent; // we want to restore the outer scope
            break;
                
        case "for-loop exit":
            var forLoopToExit;
            if (currentRepeatableBlockOfCode.type === "for-loop") {
                forLoopToExit = currentRepeatableBlockOfCode;
            } else if (currentRepeatableBlockOfCode.type === "for-loop body") {
                forLoopToExit = currentRepeatableBlockOfCode.parent;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            forLoopToExit.loopCount = message.totalLoopCount;
            currentRepeatableBlockOfCode = forLoopToExit.parent; // we want to restore the outer scope
            break;
        case "for-in-loop exit":
            var forInLoopToExit;
            if (currentRepeatableBlockOfCode.type === "for-in-loop") {
                forInLoopToExit = currentRepeatableBlockOfCode;
            } else if (currentRepeatableBlockOfCode.type === "for-in-loop body") {
                forInLoopToExit = currentRepeatableBlockOfCode.parent;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            forInLoopToExit.loopCount = message.totalLoopCount;
            currentRepeatableBlockOfCode = forInLoopToExit.parent; // we want to restore the outer scope
            break;
        case "while-loop exit":
            var whileLoopToExit;
            if (currentRepeatableBlockOfCode.type === "while-loop") {
                whileLoopToExit = currentRepeatableBlockOfCode;
            } else if (currentRepeatableBlockOfCode.type === "while-loop body") {
                whileLoopToExit = currentRepeatableBlockOfCode.parent;
            } else {
                throw {name: "Unexpected repeatable block type",
                       infringingType: newRepeatableBlock.type};
            }
            whileLoopToExit.loopCount = message.totalLoopCount;
            currentRepeatableBlockOfCode = whileLoopToExit.parent; // we want to restore the outer scope
            break;
                
        case "function declaration":
            newRepeatableBlock = {type: "function"};
            newRepeatableBlock.isRepeatableBlock = true;
            newRepeatableBlock.location = message.location;
            newRepeatableBlock.bodyLocation = message.bodyLocation;
            newRepeatableBlock.iterations = [];
            newRepeatableBlock.identifier = utils.calculateIdentifierFromLocation(message.location) + ":" + message.declarationIndex;
            newRepeatableBlock.declarationIndex = message.declarationIndex;
            newRepeatableBlock.parent = currentRepeatableBlockOfCode;
            arrayToPushTo.push(newRepeatableBlock);
            executionData.functions[newRepeatableBlock.identifier] = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
        case "function enter":
            newRepeatableBlock = {type: "function body"};
            newRepeatableBlock.body = [];
            newRepeatableBlock.argumentNames = message.argNames;
            newRepeatableBlock.argumentValues = message.argValues;
            newRepeatableBlock.argumentLocations = message.argumentLocations;
                
            var functionBlock = executionData.functions[utils.calculateIdentifierFromLocation(message.location) + ":" + message.declarationIndex];
            if (!functionBlock) {
                error = new Error("Undeclared function executed: " + JSON.stringify(message.location));
                error.functionLocation = message.location;
                throw error;
            }
            newRepeatableBlock.location = message.location;
            newRepeatableBlock.bodyLocation = functionBlock.bodyLocation;
            newRepeatableBlock.functionDeclaration = functionBlock;
            newRepeatableBlock.parent = currentRepeatableBlockOfCode;
            functionBlock.iterations.push(newRepeatableBlock);
            currentRepeatableBlockOfCode = newRepeatableBlock;
            executionData.numberOfExecutedStatements += 1;
            break;
        case "return":
            if (message.hasOwnProperty("value")) {
                arrayToPushTo.push({
                    value: message.value,
                    name: "return",
                    location: message.location,
                    type: message.type,
                    parent: currentRepeatableBlockOfCode
                });
            }
            executionData.numberOfExecutedStatements += 1;
            // fall-through
        case "function exit":
            currentRepeatableBlockOfCode = currentRepeatableBlockOfCode.parent; // we want to restore the outer scope
            break;
        case "while-loop-test":
        case "for-loop-test":
            if ((currentRepeatableBlockOfCode.type !== "while-loop body") 
                && (currentRepeatableBlockOfCode.type !== "while-loop") 
                && (currentRepeatableBlockOfCode.type !== "for-loop body") 
                && (currentRepeatableBlockOfCode.type !== "for-loop")) {
                throw new Error("Unexpected repeatable block type: " + currentRepeatableBlockOfCode.type);
            } // else
            var loopType;
            if (currentRepeatableBlockOfCode.type.indexOf("while") > -1) {
                loopType = "while";
            } else {
                loopType = "for";
            }
            currentRepeatableBlockOfCode.test = {
                value: message.value,
                result: message.result,
                name: loopType + "-condition",
                location: message.location,
                type: message.type,
                parent: currentRepeatableBlockOfCode
            };
            break;
        case "assignment":
        case "update":
        case "console log":
        case "console error":
        case "variable initilization":
        case "if-test":
        case "switch-condition":
            // report the actual values that were assigned
            arrayToPushTo.push({
                value: message.value,
                result: message.result,
                name: message.name,
                location: message.location,
                type: message.type,
                parent: currentRepeatableBlockOfCode
            });
            executionData.numberOfExecutedStatements += 1;
            break;
        case "caught exception":
        case "uncaught exception":
            arrayToPushTo.push({
                type: message.type === "caught exception" ? "caught error" : "error",
                message: message.exception.message,
                location: message.location,
                parent: currentRepeatableBlockOfCode
            });
            executionData.numberOfExecutedStatements += 1;
            if (message.type === "uncaught exception") {
                markRepeatableBlockAsHavingUncaughtException(currentRepeatableBlockOfCode);
            }
            break;
        case "catch":
            arrayToPushTo.push({
                type: "catch",
                parameter: message.exception,
                location: message.location,
                parent: currentRepeatableBlockOfCode
            });
            executionData.numberOfExecutedStatements += 1;
            break;
        }
        
        lastExecutedLineIndex = message.location.start.line;
    }
    Evaluator.setAssignmentCallback(newBitOfCodeWasExecuted);
    
    function _updateExecutionData() {
        if (codeToPreview === null) {
            return;
        }
        
        // reset execution data storage
        currentRepeatableBlockOfCode = {type: "program"};
        encounteredRepetableBlocksOfCode = {};
        executionData = {topMostExecutionContext: currentRepeatableBlockOfCode,
                         numberOfExecutedStatements: 0,
                         functions: {}};
        
        
        var codeLines = codeToPreview.split("\n");
        
        var functionName,
            functionInformation;
        linesWithGeneratedFunctions = [];
        
        // get the functions and sort them by their line index
        var functionInformationObjects = [];
        for (functionName in functionsWithParameters) {
            if (functionsWithParameters.hasOwnProperty(functionName)) {
                functionInformation = functionsWithParameters[functionName];
                functionInformation.name = functionName;
                functionInformationObjects.push(functionInformation);
            }
        }
        
        /* used to get input for functions and generate calls to those functions, but not working at the moment
        
        // the following loops do two things:
        //  1. Tell the view to provide parameters fields for the parameters of all functions
        //  2. for each function, generate a call to that function with the arguments specified in the preview (saved in parameterInformation.value below)
        functionInformationObjects.forEach(function (functionInformation, functionIndex) {
            var functionCallString = functionInformation.name + "(",
                parameters = functionInformation.parameters,
                parameterName;
            
            var comma = "";
            for (parameterName in parameters) {
                if (parameters.hasOwnProperty(parameterName)) {
                    var parameterInformation = parameters[parameterName];
                    functionCallString += comma + JSON.stringify(parameterInformation.value);
                    comma = ", ";
                    
                    liveDataView.provideInputForParameterOfFunctionInLine(parameterInformation,
                                                                          {functionName: functionInformation.name,
                                                                           name: parameterName,
                                                                           location: parameterInformation.location});
                    
                }
            }
            
            functionCallString  += ");";
            
            // add the generated function call to the code
            linesWithGeneratedFunctions.push(functionInformation.lineToInsertCall);
            var lineOfFunction = functionInformation.lineToInsertCall + functionIndex; // the index tells us how many functions calls have been added before, we need to adjust the line index by that
            codeLines.splice(lineOfFunction, 0,  functionCallString);
        });*/
        var codeToEvaluate = codeLines.join("\n");
        executionData.location = {start: {line: 0, ch: 0}, end: {line: codeToEvaluate.length, ch: 0}};
        // console.log(codeToEvaluate);
        Evaluator.runCodeAsynchronouslyAndReportResults(codeToEvaluate);
    }
    
    function _setCodeToPreview(newCodeToPreview) {
        var modifiedCodeToPreview, syntaxTree;
        modifiedCodeToPreview = newCodeToPreview;
//        if (newCodeToPreview === null) {
//            modifiedCodeToPreview = null;
//        } else {
//            modifiedCodeToPreview = newCodeToPreview.replace(/\n\n/g, "\n \n");
//        }
        
        
        // check whether it's the same code
        if (modifiedCodeToPreview === codeToPreview) { // yes, this is actually a lot faster than first trying to calculate a hashcode or something like that
            // it's the same code, we don't need to reevaluate it.
            return;
        }
        
        // otherwise it's new code, stop the old execution
        Evaluator.stopExecution();
        setExecuting(false);
        
        
        codeToPreview = modifiedCodeToPreview;
        
        // this function will be called asynchronously (but on the main thread) via setTimeout
        function evaluateCurrentCode() {
            var errorLocation = {start: {}, end: {}};
            console.log("evaluating current code");
            
            if (codeToPreview.trim().length === 0) {
                setExecuting(false);
                currentError = undefined;
                
                // reset execution data storage
                currentRepeatableBlockOfCode = {type: "program"};
                encounteredRepetableBlocksOfCode = {};
                executionData = {topMostExecutionContext: currentRepeatableBlockOfCode,
                                 numberOfExecutedStatements: 0,
                                 functions: {}};
                
                // we are done here
                return;
            }
            
            // check whether it can actually compile
            try {
                syntaxTree = esprima.parse(codeToPreview, {loc: true});
            } catch (syntaxError) {
                errorLocation.start.line = syntaxError.lineNumber - 1; //esprima starts at line 1
                errorLocation.end.line = syntaxError.lineNumber - 1; //esprima starts at line 1
                errorLocation.start.ch = syntaxError.column;
                errorLocation.end.ch = syntaxError.column + 1;
                
                // indicate that a syntax error prevents the live evaluation.
                currentError = {type: "syntaxError", error: syntaxError, message: syntaxError.message, location: errorLocation};
                return;
            }
            var canCompile = (syntaxTree !== undefined);
            
            if (canCompile) {
                setExecuting(true);
                lastExecutedLineIndex = 0;
                // reset the current error
                currentError = undefined;
                

                var esprimaResult = new EsprimaResult(syntaxTree);
                var functionNodes = esprimaResult.filter(function (treeObject) {return (treeObject.type === "FunctionDeclaration") || (treeObject.type === "FunctionExpression"); });
                
                // get the information about functions from JSLINT
                var oldFunctionsWithParameters = functionsWithParameters; // keep the old values to be able to copy them over
                functionsWithParameters = {}; // reset
                functionNodes.forEach(function (currentFunctionInCode, index, array) {
                    var functionName,
                        oldFunctionForThisName,
                        functionForThisName     = {};
                    
                    if (currentFunctionInCode.type === "FunctionDeclaration") {
                        functionName = currentFunctionInCode.id.name;
                    } else if (currentFunctionInCode.type === "FunctionExpression") {
                        if (currentFunctionInCode.parent.type === "VariableDeclarator") {
                            functionName = currentFunctionInCode.parent.id.name;
                        } else if (currentFunctionInCode.parent.type === "AssignmentExpression") {
                            functionName = currentFunctionInCode.parent.left.name;
                        } else if (currentFunctionInCode.parent.type === "CallExpression") {
                            // is already called, we don't have to do anything.
                            return;
                        }
                    }
                    oldFunctionForThisName  = oldFunctionsWithParameters[functionName];
                    functionForThisName     = {};
    
                    functionForThisName.lineIndex = currentFunctionInCode.loc.start.line - 1;
                    functionForThisName.location = {start: {line: currentFunctionInCode.loc.start.line - 1, ch: currentFunctionInCode.loc.start.column},
                                                    end:   {line: currentFunctionInCode.body.loc.end.line - 1, ch: currentFunctionInCode.body.loc.end.column}
                                                   };
                    functionForThisName.lineToInsertCall = functionForThisName.location.end.line + 1;
                    if (currentFunctionInCode.type === "FunctionExpression") {
                        // a function expression might be part of an assignment or variable initialization or function call. 
                        // In almost all of the cases, we can't insert the call directly after the function expression but have to insert it after the containing expression/declration/etc.
                        if (currentFunctionInCode.parent.type === "VariableDeclarator") {
                            functionForThisName.lineToInsertCall = (currentFunctionInCode.parent.parent.loc.end.line - 1) + 1; // take the end of the declarations (two steps up the tree) // -1 because index starts at 1, +1 because we want the next line
                        }
                    }
                    functionForThisName.parameters = {};
                    currentFunctionInCode.params.forEach(function (parameterOfFunction) {
                        var parameterForThisNameForThisFunction = oldFunctionForThisName && oldFunctionForThisName.parameters[parameterOfFunction.name];
                        if (parameterForThisNameForThisFunction === undefined) {
                            parameterForThisNameForThisFunction = {};
                        }
                        parameterForThisNameForThisFunction.location = {
                            start: {line: parameterOfFunction.loc.start.line - 1, ch: parameterOfFunction.loc.start.column}, // adjusted by -1 because the line start at 1 in eprima but 0 in the editor
                            end: {line: parameterOfFunction.loc.end.line - 1, ch: parameterOfFunction.loc.end.column} // adjusted by -1 because the values start at 1 in eprima but 0 in the editor
                        };
                        parameterForThisNameForThisFunction.line = parameterForThisNameForThisFunction.location.start.line;
                        parameterForThisNameForThisFunction.name = parameterOfFunction.name;
                        
                        functionForThisName.parameters[parameterOfFunction.name] = parameterForThisNameForThisFunction;
                    });
                    
                    functionsWithParameters[functionName] = functionForThisName;
                });
                
                // show new results
                _updateExecutionData();
                
            }
        }
        
        function evaluateNewCodeAsynchronously(codeToEvaluate) {
            window.setTimeout(function () {
                // if the code wasn't changed in between
                if (codeToEvaluate === codeToPreview) {
                    evaluateCurrentCode();
                }
            }, 20);
        }
        
        evaluateNewCodeAsynchronously(codeToPreview);
    }
    
    function setDocumentToPreview(newDocumentToPreview) {
        // reset saved functions
        functionsWithParameters = {};
        
        if ((documentToPreview !== undefined) && (documentToPreview !== null)) {
            $(documentToPreview).off("change.liveDataController");
            
            if (documentToPreview._masterEditor) {
                $(documentToPreview._masterEditor).off("cursorActivity.liveDataController");
            }
                                                       
            documentToPreview.releaseRef();
        }
        
        if (newDocumentToPreview === null) {
            documentToPreview = null;
            currentError = {type: "NoDocumentError",
                            message: "No code to evaluate. Please open a Javascript source file."};
        } else {
            if (newDocumentToPreview.getLanguage()._name !== "JavaScript") {
                documentToPreview = null;
                currentError = {type: "NotJSCodeError",
                            message: "Can only evaluate Javascript code. Please open a Javascript source file."};
            } else {
                documentToPreview = newDocumentToPreview;
            }
        }
        
        if (documentToPreview === null) {
            codeToPreview = null;
        } else {
            documentToPreview.addRef();
            $(documentToPreview).on("change.liveDataController", function () {
                _setCodeToPreview(documentToPreview.getText());
            });
            
            _setCodeToPreview(documentToPreview.getText());
            
            documentToPreview._ensureMasterEditor();
            $(documentToPreview._masterEditor).on("cursorActivity.liveDataController", selectionChanged);
        }
    }
    
    exports.setDataView = function (dataView) {
        liveDataView = dataView;
    };
    
    exports.setPreviewPane = function (newPreviewPane) {
        previewPane = newPreviewPane;
    };
    
    exports.setDocumentToPreview = setDocumentToPreview;
    
    
    // Data source methods
    
    exports.hasExecutionData = function () {
        return (executionData.topMostExecutionContext.body !== undefined) && (executionData.numberOfExecutedStatements > 0);
    };
    
    exports.getNumberOfLinesOfExecutedCode = function () {
        if ((codeToPreview === undefined) || (codeToPreview === null)) {
            return 0;
        } // else
        return codeToPreview.split("\n").length;
    };
    
    exports.getExecutionData = function () {
        return Object.create(executionData.topMostExecutionContext);
    };
    
    
    exports.isExecutingCode = function () {
        return isExecuting;
    };
    
    exports.hasError = function () {
        return currentError !== undefined;
    };
    
    exports.getError = function () {
        if (!currentError) {
            return undefined;
        }
        return currentError;
    };
    
    exports.getLastExecutedLineIndex = function () {
        return lastExecutedLineIndex;
    };
    
    
    // Delegate methods
    exports.mouseEnteredValueDisplayForValueData = function (valueData) {
        var markOptionsForValueOriginHighlight  = {className: "lc-value-origin-highlight"};
        documentToPreview._ensureMasterEditor();
        if (valueData.location) {
            if ((valueData.location.start.line === valueData.location.end.line) && (valueData.location.start.ch === valueData.location.end.ch)) {
                // start and end location are the same
                // this means, we don't now the range (this usually happens for error location)
                // simply highlight the first character at that location instead
                valueData.highlight = documentToPreview._masterEditor._codeMirror.markText(valueData.location.start, {line: valueData.location.start.line, ch: valueData.location.start.ch + 1}, markOptionsForValueOriginHighlight);
            } else {
                valueData.highlight = documentToPreview._masterEditor._codeMirror.markText(valueData.location.start, valueData.location.end, markOptionsForValueOriginHighlight);
            }
        }
    };
    
    exports.mouseLeftValueDisplayForValueData = function (valueData) {
        // we saved the highlight marker in the valueData object, so we should be able to retrieve it from there
        if (valueData.highlight) {
            valueData.highlight.clear();
        }
    };
    
    
    // start a regular update that checks whether the execution data changed and if so tells the view to update
    window.setTimeout((function () {
        var displayedExecutionData = executionData,
            displayedError = currentError,
            displayedNumberOfExecutedStatements = executionData.numberOfExecutedStatements,
            wasExecuting = isExecuting;
        
        //var lastCheck = new Date();
        var updateViewIfDataChanged = function () {
            //console.log("check interval: " + (new Date() - lastCheck));
            //lastCheck = new Date();
            if ((displayedExecutionData !== executionData)
                    || (displayedNumberOfExecutedStatements !== executionData.numberOfExecutedStatements)
                    || (displayedError !== currentError)
                    || (wasExecuting !== isExecuting)) {
                displayedExecutionData = executionData;
                displayedNumberOfExecutedStatements = executionData.numberOfExecutedStatements;
                displayedError = currentError;
                wasExecuting = isExecuting;
                
                utils.logExecutionTime("reload data: ", function () {
                liveDataView.reloadData();
                });
            }
            // schedule this function again for the next update
            window.setTimeout(updateViewIfDataChanged, 100);
        };
        return updateViewIfDataChanged;
    }()), 100); // update the view every 100ms if the data changed
});