/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, maxerr: 50 */
/*global define, $, document, window, brackets  */

/**
 * An object to hold Esprima results and allow querying them.
 */

define(function (require, exports, module) {
    "use strict";

    // Helper functions

    //returns -1 if first > second, 0 if first == second, 1 if second > first
    function locationComparison(first, second) {
        if ((first.line === second.line) && (first.column === second.column)) {
            return 0;
        } //else 
        if (first.line > second.line) {
            return -1;
        } //else 
        if ((first.line === second.line) && (first.column > second.column)) {
            return -1;
        } //else
        return 1;
    }
    
    /**
    * add a parent property to each of the tree objects pointing to its parent
    */
    function _setupTreeForTraversal(treeObject, parent) {
        treeObject.parent = parent;
        treeObject.filter = _setupTreeForTraversal.filter;
        
        if (treeObject.body instanceof Array) {
            treeObject.body.forEach(function (childObject) {
                _setupTreeForTraversal(childObject, treeObject);
            });
        } else if (treeObject.body instanceof Object) {
            _setupTreeForTraversal(treeObject.body, treeObject);
        }
        if (treeObject.declarations instanceof Array) {
            treeObject.declarations.forEach(function (childObject) {
                _setupTreeForTraversal(childObject, treeObject);
            });
        }
        if (treeObject.init instanceof Object) {
            _setupTreeForTraversal(treeObject.init, treeObject);
        }
        if (treeObject.expression instanceof Object) {
            _setupTreeForTraversal(treeObject.expression, treeObject);
        }
        if (treeObject.left instanceof Object) {
            _setupTreeForTraversal(treeObject.left, treeObject);
        }
        if (treeObject.right instanceof Object) {
            _setupTreeForTraversal(treeObject.right, treeObject);
        }
        if (treeObject.callee instanceof Object) {
            _setupTreeForTraversal(treeObject.callee, treeObject);
        }
        if (treeObject["arguments"] instanceof Array) {
            treeObject["arguments"].forEach(function (childObject) {
                _setupTreeForTraversal(childObject, treeObject);
            });
        }
        if (treeObject.test instanceof Object) {
            _setupTreeForTraversal(treeObject.test, treeObject);
        }
        if (treeObject.update instanceof Object) {
            _setupTreeForTraversal(treeObject.update, treeObject);
        }
    }
        
        /**
        * @return an array of all tree objects for which the given test function returns true
        */
    _setupTreeForTraversal.filter = function (testFunction) {
        var matchingChildrenAndSelf = [];
        var arrayOfMatchingChildrenArrays;
        // check if this should be added
        if (testFunction(this)) {
            matchingChildrenAndSelf.push(this);
        }
        
        // recursive call to gather all the children
        if (this.body instanceof Array) {
            arrayOfMatchingChildrenArrays = this.body.map(function (treeObject) {return treeObject.filter(testFunction); });
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat.apply(matchingChildrenAndSelf, arrayOfMatchingChildrenArrays);
        } else if (this.body instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.body.filter(testFunction));
        }
        if (this.declarations instanceof Array) {
            arrayOfMatchingChildrenArrays = this.declarations.map(function (treeObject) {return treeObject.filter(testFunction); });
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat.apply(matchingChildrenAndSelf, arrayOfMatchingChildrenArrays);
        }
        if (this.init instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.init.filter(testFunction));
        }
        if (this.expression instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.expression.filter(testFunction));
        }
        if (this.left instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.left.filter(testFunction));
        }
        if (this.right instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.right.filter(testFunction));
        }
        if (this.callee instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.callee.filter(testFunction));
        }
        if (this["arguments"] instanceof Array) {
            arrayOfMatchingChildrenArrays = this["arguments"].map(function (treeObject) {return treeObject.filter(testFunction); });
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat.apply(matchingChildrenAndSelf, arrayOfMatchingChildrenArrays);
        }
        if (this.test instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.test.filter(testFunction));
        }
        if (this.update instanceof Object) {
            matchingChildrenAndSelf = matchingChildrenAndSelf.concat(this.update.filter(testFunction));
        }
        return matchingChildrenAndSelf;
    };
    
    function EsprimaResult(newStructure) {
        this._structure = newStructure;
        _setupTreeForTraversal(this._structure);
    }

    EsprimaResult.EsprimaLocationFromCmLocation = function (cmLocation) {
        return { line: cmLocation.line + 1, column: cmLocation.ch };
    };

    EsprimaResult.CmLocationFromEsprimaLocation = function (esprimaLocation) {
        return { line: esprimaLocation.line - 1, ch: esprimaLocation.column };
    };

    // not needed afaik, just if subclassing another class and then it should be "= new SuperClass()" 
    //EsprimaResult.prototype                 = new EsprimaResult(); 
    //EsprimaResult.prototype.constructor     = EsprimaResult;

    // EsprimaResult.prototype._structure      = {}; // useless if this._structure is always set in the constructor

    EsprimaResult.prototype.getStructure = function () {
        return this._structure;
    };

    EsprimaResult.prototype.setStructure = function (newStructure) {
        this._structure = newStructure;
    };

    EsprimaResult.prototype.filter = function (testFunction) {
        return this._structure.filter(testFunction);
    };
    
    EsprimaResult.prototype.tokensInRange = function (start, end, typeFilter) {
        start   = EsprimaResult.EsprimaLocationFromCmLocation(start);
        end     = EsprimaResult.EsprimaLocationFromCmLocation(end);

        var result = [];
        var i;
        for (i = 0; i < this._structure.tokens.length; i++) {
            var token = this._structure.tokens[i];

            if (typeFilter !== null) {
                if (token.type !== typeFilter) {
                    continue;
                }
            }

            if ((locationComparison(token.loc, start) <= 0) && (locationComparison(token.loc, end) >= 0)) {
                result.push(token);
            }
        }

        return result;
    };

    EsprimaResult.prototype.elementsInRangeByType = function (start, end, typeFilter) {
        start   = EsprimaResult.EsprimaLocationFromCmLocation(start);
        end     = EsprimaResult.EsprimaLocationFromCmLocation(end);

        var result = this.filter(function () {
            if (this.type !== typeFilter) {
                return false;
            }

            if ((locationComparison(this.loc.start, start) <= 0) && (locationComparison(this.loc.end, end) >= 0)) {
                return true;
            }

            return false;
        });

        return result;
    };

    EsprimaResult.prototype.blocklevelElementEndingInPosition = function (position) {
        var compareLineOnly = (position.column === undefined);

        //Esprima line indexes start at 1
        position = EsprimaResult.EsprimaLocationFromCmLocation(position);

        var result = this.filter(function () {
            //test if we have a block level element at all
            if (this.body === undefined) {
                return false;
            }

            //test if end is correct
            if (compareLineOnly) {
                return (this.loc.end.line === position.line);
            } //else
            return ((this.loc.end.line === position.line) && (this.loc.end.column === position.column));
        });

        return (result.length > 0) ? result[0] : undefined;
    };

    EsprimaResult.prototype.functionForPosition = function (position) {
        var compareLineOnly = (position.column === undefined);

        //Esprima line indexes start at 1
        position = EsprimaResult.EsprimaLocationFromCmLocation(position);

        var result = this.filter(function () {
            if (this.type !== "FunctionDeclaration") {
                return false;
            }

            var comparisonResult = (this.loc.start.line <= position.line) && (this.loc.end.line >= position.line);

            if (!compareLineOnly) {
                comparisonResult = comparisonResult && (this.loc.start.column <= position.column) && (this.loc.end.column >= position.column);
            }

            return comparisonResult;
        });

        return (result.length > 0) ? result[0] : undefined;
    };
    
    
    

    return EsprimaResult;

});