//
//  FABProjectsTableViewController.m
//  Fabiji
//
//  Created by Zhao He on 2/15/12.
//  Copyright (c) 2012 Media Computing Group, RWTH Aachen University. All rights reserved.
//

#import "FABProjectsSectionViewController.h"
#import "FABTool.h"
#import "FABMaterial.h"
#import "FABLicense.h"
#import "FABTag.h"
#import "TITokenFieldView+Utility.h"
#import "NSString+Utility.h"
#import "NSArray+NameContents.h"

@interface FABProjectsSectionViewController()

typedef enum {
    kLoadTools,
    kLoadMaterials,
    kLoadLicenses,
    kLoadTags,
    kCreateTag,
    kDeleteTag,
    kCreateTool,
    kDeleteTool,
    kCreateMaterial,
    kDeleteMaterial,
    kUpdateProject,
} FABRequestMethodType;

@property (nonatomic, assign) FABRequestMethodType requestMethod;
@property (retain, nonatomic) TITokenFieldView *tagsTokenFieldView;
@property (retain, nonatomic) TITokenFieldView *toolsTokenFieldView;
@property (retain, nonatomic) TITokenFieldView *materialsTokenFieldView;

@end

@implementation FABProjectsSectionViewController

@synthesize requestMethod = _requestMethod;
@synthesize tagsTokenFieldView = _tagsTokenFieldView;
@synthesize toolsTokenFieldView = _toolsTokenFieldView;
@synthesize materialsTokenFieldView = _materialsTokenFieldView;
@synthesize cellTextHeight = _cellTextHeight;
@synthesize currentProject = _currentProject;
@synthesize availableTools = _availableTools;
@synthesize availableMaterials = _availableMaterials;
@synthesize availableLicenses = _availableLicenses;
@synthesize availableTags = _availableTags;
@synthesize descriptionTextView = _descriptionTextView;
@synthesize toolsTextField = _toolsTextField;
@synthesize materialsTextField = _materialsTextField;
@synthesize licenseTextField = _licenseTextField;
@synthesize tagsTextField = _tagsTextField;
@synthesize pickerPopoverController = _pickerPopoverController;
@synthesize descriptionCell = _descriptionCell;
@synthesize toolsCell = _toolsCell;
@synthesize materialsCell = _materialsCell;
@synthesize licenseCell = _licenseCell;
@synthesize tagsCell = _tagsCell;
@synthesize tagsHintLabel = _tagsHintLabel;

- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) { 
        // Initial rowHeight for cells
        self.cellTextHeight = 44.0;
    }
    return self;
}


- (UITextView *)descriptionTextView
{
    _descriptionTextView = (UITextView *)[self.descriptionCell viewWithTag:2];
    return _descriptionTextView;
}

- (TITokenField *)toolsTextField
{
    _toolsTextField = self.toolsTokenFieldView.tokenField;
    return _toolsTextField;
}

- (TITokenField *)materialsTextField
{
    _materialsTextField = self.materialsTokenFieldView.tokenField;
    return _materialsTextField;
}

- (UITextField *)licenseTextField
{
    _licenseTextField = (UITextField *)[self.licenseCell viewWithTag:2];
    return _licenseTextField;
}

- (TITokenField *)tagsTextField
{
    _tagsTextField = self.tagsTokenFieldView.tokenField;
    return _tagsTextField;
}


- (void)dealloc 
{
    [_currentProject release];
    [_availableTools release];
    [_availableLicenses release];
    [_descriptionTextView release];
    [_toolsTextField release];
    [_materialsTextField release];
    [_licenseTextField release];
    [_tagsTextField release];
    [_pickerPopoverController release];
    [_toolsCell release];
    [_descriptionCell release];
    [_materialsCell release];
    [_licenseCell release];
    [_tagsCell release];
    [_tagsHintLabel release];
    [super dealloc];
}

#pragma mark RestKit

- (void)loadTools
{
    // GET /tools
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:@"/tools" delegate:self];
    self.requestMethod = kLoadTools;
}

- (void)loadMaterials
{
    // GET /materials
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:@"/materials" delegate:self];
    self.requestMethod = kLoadMaterials;
}

- (void)loadLicenses
{
    // GET /licenses
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:@"/licenses" delegate:self];
    self.requestMethod = kLoadLicenses;
}

- (void)loadTags
{
    // Get to /tags
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:@"/tags" delegate:self];
    self.requestMethod = kLoadTags;
}

- (void)createTag:(NSString *)tagString atProjectWithID:(NSNumber *)pid
{
    FABTag *tag = [[[FABTag alloc] init] autorelease];
    tag.name = tagString;
    [[RKObjectManager sharedManager] postObject:tag usingBlock:^(RKObjectLoader* loader){
        NSString *request = [NSString stringWithFormat:@"/projects/%@/tags", pid];
        loader.resourcePath = request;
        loader.delegate = self;
    }];
    self.requestMethod = kCreateTag;
}

- (void)deleteTagWithID:(NSNumber *)tid
{
    FABTag *tag = [[[FABTag alloc] init] autorelease];
    tag.identifier = tid;
    
    // DELETE to /tags/tid
    [ [RKObjectManager sharedManager] deleteObject:tag delegate:self];
    self.requestMethod = kDeleteTag;
}

- (void)createTool:(NSString *)toolString atProjectWithID:(NSNumber *)pid
{
    FABTool *tool = [[[FABTool alloc] init] autorelease];
    tool.name = toolString;
    [[RKObjectManager sharedManager] postObject:tool usingBlock:^(RKObjectLoader* loader){
        NSString *request = [NSString stringWithFormat:@"/projects/%@/tools", pid];
        loader.resourcePath = request;
        loader.delegate = self;
    }];
    self.requestMethod = kCreateTool;
}

- (void)deleteToolWithID:(NSNumber *)tid
{
    FABTool *tool = [[[FABTool alloc] init] autorelease];
    tool.identifier = tid;
    
    // DELETE to /tools/tid
    [ [RKObjectManager sharedManager] deleteObject:tool delegate:self];
    self.requestMethod = kDeleteTool;
}

- (void)createMaterial:(NSString *)materialString atProjectWithID:(NSNumber *)pid
{
    FABMaterial *material = [[[FABMaterial alloc] init] autorelease];
    material.name = materialString;
    [[RKObjectManager sharedManager] postObject:material usingBlock:^(RKObjectLoader* loader){
        NSString *request = [NSString stringWithFormat:@"/projects/%@/materials", pid];
        loader.resourcePath = request;
        loader.delegate = self;
    }];
    self.requestMethod = kCreateMaterial;
}

- (void)deleteMaterialWithID:(NSNumber *)mid
{
    FABMaterial *material = [[[FABMaterial alloc] init] autorelease];
    material.identifier = mid;
    
    // DELETE to /materials/mid
    [ [RKObjectManager sharedManager] deleteObject:material delegate:self];
    self.requestMethod = kDeleteMaterial;
}

- (void)updateCurrentProjectWithLicense:(FABLicense *)newLicense
{
    NSString *deleteRequest = [NSString stringWithFormat:@"/licenses/%@/projects/%@", self.currentProject.licenseOfProject.identifier, self.currentProject.identifier];
    [[RKClient sharedClient] delete:deleteRequest delegate:self];
    NSString *postRequest = [NSString stringWithFormat:@"/licenses/%@/projects/%@", newLicense.identifier, self.currentProject.identifier];
    [[RKClient sharedClient] post:postRequest params:nil delegate:self];
    
    self.requestMethod = kUpdateProject;
}


- (void)request:(RKRequest*)request didLoadResponse:(RKResponse*)response 
{
    //NSLog(@"Response: %@", [response bodyAsString]);
    NSLog(@"Status: %d", [response statusCode]);
    if ([response isOK]) {
        NSLog(@"response isOK.");
    }
    else if([response isCreated]) {
        NSLog(@"response isCreated.");
    }
}

- (void)objectLoader:(RKObjectLoader*)objectLoader didLoadObjects:(NSArray *)objects 
{
    //NSLog(@"objects: %@", objects);
    if ([objectLoader.resourcePath isEqualToString:@"/tools"]) 
    {        
        // Set up source array
        self.availableTools = [objects nameContentsArray];
        self.toolsTokenFieldView.sourceArray = self.availableTools;
    }
    if ([objectLoader.resourcePath isEqualToString:@"/materials"])
    {
        // Set up source array
        self.availableMaterials = [objects nameContentsArray];
        self.materialsTokenFieldView.sourceArray = self.availableMaterials;
    }
    if ([objectLoader.resourcePath isEqualToString:@"/licenses"])
    {
        self.availableLicenses = [objects nameContentsArray];
    }
    if ([objectLoader.resourcePath isEqualToString:@"/tags"]) 
    {
        // Set up source array
        self.availableTags = [objects nameContentsArray];
        self.tagsTokenFieldView.sourceArray = self.availableTags;
    }
    
    if (self.requestMethod == kCreateTag)
    {
        [self.currentProject.tagsOfProject addObject:[objects objectAtIndex:0]];
    }
    if (self.requestMethod == kCreateTool)
    {
        [self.currentProject.toolsOfProject addObject:[objects objectAtIndex:0]];
    }
    if (self.requestMethod == kCreateMaterial)
    {
        [self.currentProject.materialsOfProject addObject:[objects objectAtIndex:0]];
    }
    if (self.requestMethod == kDeleteTag)
    {
        [self.currentProject.tagsOfProject removeObject:[objects objectAtIndex:0]];
    }
    if (self.requestMethod == kDeleteTool)
    {
        [self.currentProject.toolsOfProject removeObject:[objects objectAtIndex:0]];
    }
    if (self.requestMethod == kDeleteMaterial)
    {
        [self.currentProject.materialsOfProject removeObject:[objects objectAtIndex:0]];
    }
    

}

- (void)objectLoader:(RKObjectLoader*)objectLoader didFailWithError:(NSError*)error {
	UIAlertView* alert = [[[UIAlertView alloc] initWithTitle:@"Error" message:[error localizedDescription] delegate:nil cancelButtonTitle:@"OK" otherButtonTitles:nil] autorelease];
	[alert show];
	NSLog(@"Hit error: %@", error);
}

#pragma mark - View lifecycle

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    // Load available tools, materials, licenses and tags for users to select from
    [self loadTools];
    [self loadMaterials];
    [self loadLicenses];
    [self loadTags];
    
    
    // Create tagsTokenView
    UIView *tagsHolderView = [self.tagsCell viewWithTag:2];
    self.tagsTokenFieldView = [[TITokenFieldView alloc] initWithFrame:tagsHolderView.frame];
    self.tagsTokenFieldView.tokenField.delegate = self;
    [self.tagsTokenFieldView setupWithParentView:self.tagsCell sourceArray:nil placeHolderView:self.tagsHintLabel];
    
    // Create toolsTokenView
    UIView *toolsHolderView = [self.toolsCell viewWithTag:2];
    self.toolsTokenFieldView = [[TITokenFieldView alloc] initWithFrame:toolsHolderView.frame];
    self.toolsTokenFieldView.tokenField.delegate = self;
    [self.toolsTokenFieldView setupWithParentView:self.toolsCell sourceArray:nil placeHolderView:nil];

    // Create materialsTokenView
    UIView *materialsHolderView = [self.materialsCell viewWithTag:2];
    self.materialsTokenFieldView = [[TITokenFieldView alloc] initWithFrame:materialsHolderView.frame];
    self.materialsTokenFieldView.tokenField.delegate = self;
    [self.materialsTokenFieldView setupWithParentView:self.materialsCell sourceArray:nil placeHolderView:nil];



    // Set background color to clear
    if ([self.tableView respondsToSelector:@selector(backgroundView)]) 
    {
        self.tableView.backgroundView = nil;
    }
}

- (void)viewDidUnload
{
    [self setPickerPopoverController:nil];
    [self setToolsCell:nil];
    [self setDescriptionCell:nil];
    [self setMaterialsCell:nil];
    [self setLicenseCell:nil];
    [self setTagsCell:nil];
    [self setTagsHintLabel:nil];
    [[[[RKObjectManager sharedManager] client] requestQueue] cancelRequestsWithDelegate:self];
    [super viewDidUnload];
}

- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    // Reload available items
    [self loadTools];
    [self loadMaterials];
    [self loadLicenses];
    [self loadTags];
}


- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	return YES;
}


#pragma mark - Managing project sections

- (void)setCurrentProject:(FABProject *)newProject
{
    if (_currentProject != newProject) {
        [_currentProject release]; 
        _currentProject = [newProject retain]; 
        
        // Update the section fields
        [self configureView];
    }
}

- (void)setupTokenField:(TITokenField *)tokenField withArray:(NSArray*)array hintLable:(UILabel *)hintLable
{
    // Clear text field in tags
    tokenField.text = @"";
    hintLable.hidden = NO;
    
    // Remove old tokens
    // removeToken: method also removes from super view
    for (TIToken *token in tokenField.tokens) {
        [tokenField removeToken:token];
    }
    
    // Add new tokens
    if ([array count]) 
    {
        NSArray *nameContents = [array nameContentsArray];
        for (NSString *str in nameContents) {
            [tokenField addTokenWithTitle:str andFocus:NO];
        }
        
        // Hide tagsHintLabel if there are tags
        if (hintLable.hidden == NO) {
            hintLable.hidden = YES;
        }
    }
}

- (void)configureView
{    
    // Update fields
    self.descriptionTextView.text = self.currentProject.descriptionOfProject;
    self.licenseTextField.text = self.currentProject.licenseOfProject.name;
    
    [self setupTokenField:self.tagsTextField withArray:self.currentProject.tagsOfProject hintLable:self.tagsHintLabel];
    [self setupTokenField:self.toolsTextField withArray:self.currentProject.toolsOfProject hintLable:nil];
    [self setupTokenField:self.materialsTextField withArray:self.currentProject.materialsOfProject hintLable:nil];
    
    // Update cell height programmatically
    [self updateHeightOfTableViewWithTextView:self.descriptionTextView];
}


#pragma mark - Table view data source

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return 5;
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return 1;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    NSArray *cells = [NSArray arrayWithObjects:self.descriptionCell, self.toolsCell, self.materialsCell, 
                      self.licenseCell, self.tagsCell, nil];
    
    return [cells objectAtIndex:indexPath.section];
}


- (BOOL)tableView:(UITableView *)tableView canMoveRowAtIndexPath:(NSIndexPath *)indexPath
{
    // Return NO if you do not want the item to be re-orderable.
    return YES;
}


#pragma mark - Table view delegate

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    // Select a list of licenses from popover
    if (indexPath.section == 3) 
    {
        // Create picker
        FABPickerViewController *picker = [[FABPickerViewController alloc] initWithTitle:nil TableViewContents:self.availableLicenses icons:nil];
        picker.delegate = self;
        
        // Pop over available licenses
        self.pickerPopoverController = [[[UIPopoverController alloc] initWithContentViewController:picker] autorelease];
        self.pickerPopoverController.popoverContentSize = CGSizeMake(320, 44 * [self.availableLicenses count]);
        CGRect rect = [tableView convertRect:[tableView rectForRowAtIndexPath:indexPath] toView:tableView];         
        [self.pickerPopoverController presentPopoverFromRect:rect inView:tableView permittedArrowDirections:UIPopoverArrowDirectionAny animated:YES];
        [picker release];
    }
}


#pragma mark - FABPickerViewController delegation

- (void)pickerRowSelected:(int)row
{ 
    // Update text field selected license
    NSString *selectedText = [self.availableLicenses objectAtIndex:row];
    self.licenseTextField.text = selectedText;
    
    // Update currentProject
    FABLicense *newLicense = [[[FABLicense alloc] init] autorelease];
    newLicense.identifier = [NSNumber numberWithInt:row+1];
    newLicense.name = selectedText;
    self.currentProject.licenseOfProject = newLicense;
    
    // Update with server
    [self updateCurrentProjectWithLicense:newLicense];
    
    [self.pickerPopoverController dismissPopoverAnimated:YES];
    self.pickerPopoverController = nil;
}


#pragma mark - UITextFiled delegations

- (void)textFieldDidBeginEditing:(UITextField *)textField
{
    // Hide tagsHintLabel if there are tags
    if (self.tagsHintLabel.hidden == NO) {
        self.tagsHintLabel.hidden = YES;
    }
}


- (void)textFieldDidEndEditing:(UITextField *)textField
{
    if (textField == self.toolsTextField) 
    {
        NSArray *toolNamesOfProject = [self.currentProject.toolsOfProject nameContentsArray];
        
        // POST new tool if it doesn't exist in current project
        for (NSString *toolString in self.toolsTextField.tokenTitles) {
            if ([toolString isInArrayofStrings:toolNamesOfProject] == NO) {
                [self createTool:toolString atProjectWithID:self.currentProject.identifier];
            }
            
            // Update available tools
            if ([toolString isInArrayofStrings:self.availableTools] == NO) {
                [self.availableTools addObject:toolString];
            }
        }
        
        // DELETE existing tool if it is removed from textField
        for (FABTool *tool in self.currentProject.toolsOfProject) {
            if ([tool.name isInArrayofStrings: self.toolsTextField.tokenTitles] == NO) {
                [self deleteToolWithID:tool.identifier];
            }
        }
    }
    
    else if (textField == self.materialsTextField) 
    {
        NSArray *materialNamesOfProject = [self.currentProject.materialsOfProject nameContentsArray];
        
        // POST new material if it doesn't exist in current project
        for (NSString *materialString in self.materialsTextField.tokenTitles) {
            if ([materialString isInArrayofStrings:materialNamesOfProject] == NO) {
                [self createMaterial:materialString atProjectWithID:self.currentProject.identifier];
            }
            
            // Update available materials
            if ([materialString isInArrayofStrings:self.availableMaterials] == NO) {
                [self.availableMaterials addObject:materialString];
            }
        }
        
        // DELETE existing material if it is removed from textField
        for (FABMaterial *material in self.currentProject.materialsOfProject) {
            if ([material.name isInArrayofStrings: self.materialsTextField.tokenTitles] == NO) {
                [self deleteMaterialWithID:material.identifier];
            }
        }
    }
    
    else if (textField == self.tagsTextField) 
    {    
        NSArray *tagNamesOfProject = [self.currentProject.tagsOfProject nameContentsArray];
        
        // POST new tag if it doesn't exist in current project
        for (NSString *tagString in self.tagsTextField.tokenTitles) {
            if ([tagString isInArrayofStrings:tagNamesOfProject] == NO) {
                [self createTag:tagString atProjectWithID:self.currentProject.identifier];
            }
            
            // Update available tags
            if ([tagString isInArrayofStrings:self.availableTags] == NO) {
                [self.availableTags addObject:tagString];
            }
        }
        
        // DELETE existing tag if it is removed from textField
        for (FABTag *tag in self.currentProject.tagsOfProject) {
            if ([tag.name isInArrayofStrings: self.tagsTextField.tokenTitles] == NO) {
                [self deleteTagWithID:tag.identifier];
            }
        }
        
        // Show tagsHintLabel if there is no tag
        if (self.tagsTokenFieldView.tokenTitles.count == 0) {
            self.tagsHintLabel.hidden = NO;
        }
    }
}


#pragma mark - Autosize UITextView in UITableViewCell

- (void)textViewDidChange:(UITextView *)textView
{
    [self updateHeightOfTableViewWithTextView:textView];
    
    if (textView == self.descriptionTextView) 
    {
        self.currentProject.descriptionOfProject = textView.text;
        //[FABPlistHelper plistUpdateProjectAtIndexPath:self.currentProjectIndexPath WithObject:self.currentProject.descriptionOfProject ForKey:@"description"];
    }
}

- (void)updateHeightOfTableViewWithTextView:(UITextView *)textView
{
    CGFloat newCellTextHeight = [textView contentSize].height;
    
    if (newCellTextHeight != self.cellTextHeight)
    {
        self.cellTextHeight = newCellTextHeight;
        BOOL animationsEnabled = [UIView areAnimationsEnabled];
        [UIView setAnimationsEnabled:NO];
        [self.tableView beginUpdates];
        [self.tableView endUpdates];
        [UIView setAnimationsEnabled:animationsEnabled];
    }
}

- (CGFloat)tableView:(UITableView *)tableView heightForRowAtIndexPath:(NSIndexPath *)indexPath
{   
    const CGFloat minimumHeight = 44.0;
    CGFloat heightForRow = minimumHeight;
    
    //int kYourTableViewsNumberOfRows = 3;
    
    if ([indexPath section] == 0)
    {
        //CGFloat tableViewHeight = [tableView bounds].size.height;
        //heightForRow = tableViewHeight - ((kYourTableViewsNumberOfRows - 1) * heightForRow);
        heightForRow = self.cellTextHeight;
        
        // make sure row height is always larger than that of one row
        if (heightForRow < minimumHeight) 
        {
            heightForRow = minimumHeight;
        }
    }
    return heightForRow;
}


@end
