//
//  FABProjectsListViewController.m
//  Fabiji
//
//  Created by Zhao He on 06/02/2012.
//  Copyright (c) 2012 Media Computing Group, RWTH Aachen University. All rights reserved.
//

#import "FABProjectsListViewController.h"
#import "FABTool.h"
#import "FABMaterial.h"
#import "FABTag.h"

@interface FABProjectsListViewController()

typedef enum {
    kLoadProjectsOfUser,
    kLoadProject,
    kCreateProject,
    kCreateTag,
} FABRequestMethodType;

@property (nonatomic, retain) FABProject *selectedProject;
@property (nonatomic, assign) FABRequestMethodType requestMethod;
@property (nonatomic, retain) NSArray *myNewTools;
@property (nonatomic, retain) NSArray *myNewMaterials;
@property (nonatomic, retain) NSArray *myNewTags;
@property (nonatomic, retain) NSNumber *myNewProjectID;
@property (nonatomic, retain) NSNumber *myNewLicenseID;

- (void)loadProjectsOfUserWithID:(NSNumber *)uid;
- (void)createProject:(FABProject *)project atUserWithID:(NSNumber *)uid;
- (void)configureView;
- (IBAction)addButtonTapped:(id)sender;
- (IBAction)editButtonTapped:(id)sender;

@end


@implementation FABProjectsListViewController

@synthesize myNewLicenseID = _myNewLicenseID;
@synthesize myNewProjectID = _myNewProjectID;
@synthesize myNewTools = _myNewTools;
@synthesize myNewMaterials = _myNewMaterials;
@synthesize myNewTags = _myNewTags;
@synthesize requestMethod = _requestMethod;
@synthesize selectedProject = _selectedProject;
@synthesize currentUser = _currentUser;
@synthesize projects = _projects;
@synthesize detailViewController = _detailViewController;


- (FABProjectsDetailViewController *)detailViewController
{
    id dvc = [self.splitViewController.viewControllers lastObject];
    if (![dvc isKindOfClass:[FABProjectsDetailViewController class]]) {
        dvc = nil;
    }
    return dvc;
}

- (void)dealloc 
{
    [_selectedProject release];
    [_currentUser release];
    [_projects release];
    [_detailViewController release];
    [super dealloc];
}


#pragma mark - RestKit

- (void)loadProjectWithID:(NSNumber *)pid
{
    self.requestMethod = kLoadProject;
    
    // GET to /projects/pid
    NSString *request = [NSString stringWithFormat:@"/projects/%@", pid];
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:request delegate:self];
}

- (void)loadProjectsOfUserWithID:(NSNumber *)uid
{
    self.requestMethod = kLoadProjectsOfUser;
    
    // GET /projects/pid
    NSString *request = [NSString stringWithFormat:@"/users/%@/projects", uid];
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:request delegate:self];
}

- (void)createProject:(FABProject *)project atUserWithID:(NSNumber *)uid
{
    self.requestMethod = kCreateProject;
    
    // POST to /users/uid/projects
    [[RKObjectManager sharedManager] postObject:project usingBlock:^(RKObjectLoader* loader){
        loader.resourcePath = [NSString stringWithFormat:@"/users/%@/projects", uid];
        loader.delegate = self;
    }];
}

- (void)createTool:(NSString*)toolName atProjectWithID:(NSNumber *)pid
{
    FABTool *tool = [[[FABTool alloc] init] autorelease];
    tool.name = toolName;
    [[RKObjectManager sharedManager] postObject:tool usingBlock:^(RKObjectLoader* loader){
        NSString *request = [NSString stringWithFormat:@"/projects/%@/tools", pid];
        loader.resourcePath = request;
        loader.delegate = self;
    }];
}


- (void)createMaterial:(NSString*)materialName atProjectWithID:(NSNumber *)pid
{    
    FABMaterial *material = [[[FABMaterial alloc] init] autorelease];
    material.name = materialName;
    [[RKObjectManager sharedManager] postObject:material usingBlock:^(RKObjectLoader* loader){
        NSString *request = [NSString stringWithFormat:@"/projects/%@/materials", pid];
        loader.resourcePath = request;
        loader.delegate = self;
    }];
}

- (void)createTag:(NSString*)tagName atProjectWithID:(NSNumber *)pid
{
    self.requestMethod = kCreateTag;
    
    FABTag *tag = [[[FABTag alloc] init] autorelease];
    tag.name = tagName;
    [[RKObjectManager sharedManager] postObject:tag usingBlock:^(RKObjectLoader* loader){
        NSString *request = [NSString stringWithFormat:@"/projects/%@/tags", pid];
        loader.resourcePath = request;
        loader.delegate = self;
    }];
}

- (void)attachLicenseWithID:(NSNumber *)lid atProjectWithID:(NSNumber *)pid
{
    NSString *request = [NSString stringWithFormat:@"/licenses/%@/projects/%@", lid, pid];
    [[RKClient sharedClient] post:request params:nil delegate:self];
}

- (void)deleteProject:(FABProject *)project
{
    // DELETE to /projects/uid
    [ [RKObjectManager sharedManager] deleteObject:project delegate:self];
}



//- (void)request:(RKRequest*)request didLoadResponse:(RKResponse*)response 
//{
//    NSLog(@"Response: %@", [response bodyAsString]);
//    NSLog(@"Status: %d", [response statusCode]);
//    if ([response isOK]) {
//        NSLog(@"response isOK.");
//    }
//    else if([response isCreated]) {
//        NSLog(@"response isCreated.");
//    }
//}

- (void)objectLoader:(RKObjectLoader*)objectLoader didLoadObjects:(NSArray *)objects 
{
	NSLog(@"Loaded object: %@", objects);
    NSLog(@"resourcePath: %@", objectLoader.resourcePath);
    
    // kLoadProjectsOfUser
    if (objectLoader.method == RKRequestMethodGET && [objectLoader.resourcePath hasPrefix:@"/users/"]) 
    {
        self.projects = [NSMutableArray arrayWithArray:objects];
        [self.tableView reloadData];
        
        // Select the first project if any
        if (self.projects.count) {
            [self selectRow:0];
        }
        
        // Update detailViewController and hide its contents if there is no project in current user
        else 
        {
            [self.detailViewController.view setNeedsDisplay];
            self.detailViewController.scrollView.hidden = YES;
            self.detailViewController.currentUser = self.currentUser;
        }


    }
    
    // kLoadProject
    else if (objectLoader.method == RKRequestMethodGET && [objectLoader.resourcePath hasPrefix:@"/projects/"])
    {
        if (self.detailViewController) 
        {
            self.detailViewController.currentUser = self.currentUser;
            self.detailViewController.currentProject = [objects objectAtIndex:0];
        }
    }
    
    // Created a new project
    else if (objectLoader.method == RKRequestMethodPOST && [objectLoader.resourcePath hasPrefix:@"/users/"])
    {
        FABProject *newProject = [objects objectAtIndex:0];
        self.myNewProjectID = [NSNumber numberWithInt:newProject.identifier.intValue];
        
        // Add new project to local variable
        [self.projects addObject:newProject];
        
        // Add materials to server
        for (NSString *material in self.myNewMaterials) {
            [self createMaterial:material atProjectWithID:self.myNewProjectID];
        }
        
        // Attach license to server
        [self attachLicenseWithID:self.myNewLicenseID atProjectWithID:self.myNewProjectID];
        
        // Add tags to server
        for (NSString *tag in self.myNewTags) {
            [self createTag:tag atProjectWithID:self.myNewProjectID];
        }
        
        // Add tools to server
        for (NSString *tool in self.myNewTools) {
            [self createTool:tool atProjectWithID:self.myNewProjectID];
        }
        
        // Update tableView
        [self.tableView reloadData];
        
        // Select the newly added project
        int lastRow = self.projects.count - 1;
        [self selectRow:lastRow];
    }
}

- (void)objectLoader:(RKObjectLoader*)objectLoader didFailWithError:(NSError*)error {
	UIAlertView* alert = [[[UIAlertView alloc] initWithTitle:@"Error" message:[error localizedDescription] delegate:nil cancelButtonTitle:@"OK" otherButtonTitles:nil] autorelease];
	[alert show];
	NSLog(@"Hit error: %@", error);
}


#pragma mark - Managing the detail item

- (void)selectRow:(int)row
{
    NSIndexPath *indexPath = [NSIndexPath indexPathForRow:row inSection:0];
    [[self.tableView delegate] tableView:self.tableView didSelectRowAtIndexPath:indexPath];
}

- (void)setCurrentUser:(FABUser *)newUser
{
    if (_currentUser != newUser) {
        [_currentUser release]; 
        _currentUser = [newUser retain]; 
        
        // Update the view.
        [self configureView];
    }
}

- (void)configureView
{ 
    NSLog(@"current user id: %@", self.currentUser.identifier);
    self.title = self.currentUser.fullname;
    [self loadProjectsOfUserWithID:self.currentUser.identifier];
    
    if (self.currentUser == nil) {
        self.navigationItem.rightBarButtonItem.enabled = NO;
        self.navigationItem.leftBarButtonItem.enabled = NO;
    }
    else {
        self.navigationItem.rightBarButtonItem.enabled = YES;
        self.navigationItem.leftBarButtonItem.enabled = YES;
    }
}


#pragma mark - IBActions


// Override tableView method
- (void)setEditing:(BOOL)editing animated:(BOOL)animated
{
    if(self.editing)
	{
		[super setEditing:NO animated:YES]; 
		[self.tableView reloadData];
        
        self.navigationItem.leftBarButtonItem = nil;
        UIBarButtonItem *editButton = [[[UIBarButtonItem alloc] initWithTitle:@"Edit" style:UIBarButtonItemStylePlain target:self action:@selector(editButtonTapped:)] autorelease];
        self.navigationItem.leftBarButtonItem = editButton;
	}
	else
	{
		[super setEditing:YES animated:YES]; 
		[self.tableView reloadData];
        
        self.navigationItem.leftBarButtonItem = nil;
        UIBarButtonItem *doneButton = [[[UIBarButtonItem alloc] initWithBarButtonSystemItem:UIBarButtonSystemItemDone target:self action:@selector(doneButtonTapped:)] autorelease];
        self.navigationItem.leftBarButtonItem = doneButton;
	}
}

- (IBAction)addButtonTapped:(id)sender
{
    NSLog(@"addButtonTapped:");
    
    FABProjectGuideStep1ViewController *step1ViewController = [[FABProjectGuideStep1ViewController alloc] init];
    
    // Push step one with FlipFromLeft animation
    [UIView beginAnimations:nil context:NULL];
    [UIView setAnimationDuration:.4];
    [UIView setAnimationBeginsFromCurrentState:YES];        
    [UIView setAnimationTransition:UIViewAnimationTransitionFlipFromLeft forView:self.splitViewController.navigationController.view cache:NO];
    [self.splitViewController.navigationController pushViewController:step1ViewController animated:NO];
    [UIView commitAnimations];
    
    [step1ViewController release];
}

- (IBAction)editButtonTapped:(id)sender
{
    [self setEditing:YES animated:YES];
}

- (IBAction)doneButtonTapped:(id)sender
{
    [self setEditing:NO animated:YES];
}

#pragma mark - Public methods

- (void)addNewProject:(FABProject *)project
{
    self.myNewTools = [NSArray arrayWithArray:project.toolsOfProject];
    self.myNewMaterials = [NSArray arrayWithArray:project.materialsOfProject];
    self.myNewTags = [NSArray arrayWithArray:project.tagsOfProject];
    self.myNewLicenseID = [NSNumber numberWithInt:project.licenseOfProject.identifier.intValue];
    
    // POST new project to server
    [self createProject:project atUserWithID:self.currentUser.identifier];
}


#pragma mark - View lifecycle

- (void)viewDidLoad
{
    [super viewDidLoad];
        
    // Add "+" button
    UIBarButtonItem *addButton = [[[UIBarButtonItem alloc] initWithBarButtonSystemItem:UIBarButtonSystemItemAdd target:self action:@selector(addButtonTapped:)] autorelease];
    self.navigationItem.rightBarButtonItem = addButton;
    
    // Add "Edit" button
    UIBarButtonItem *editButton = [[[UIBarButtonItem alloc] initWithTitle:@"Edit" style:UIBarButtonItemStylePlain target:self action:@selector(editButtonTapped:)] autorelease];
    self.navigationItem.leftBarButtonItem = editButton;
    
    // Hide the back button
    [self.navigationItem setHidesBackButton:YES animated:YES];
    
    [self configureView];
}

- (void)viewDidUnload 
{
    [[[[RKObjectManager sharedManager] client] requestQueue] cancelRequestsWithDelegate:self];
    [super viewDidUnload];
}

- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    return YES;
}


#pragma mark - UITableView delegation

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return 1;
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return [self.projects count];
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
    self.selectedProject = [self.projects objectAtIndex:indexPath.row];
    [self loadProjectWithID:self.selectedProject.identifier];
    
    // Add checkmark for selected cell
    for (int i = 0; i <= self.projects.count-1; i++) 
    {
        NSIndexPath *ip = [NSIndexPath indexPathForRow:i inSection:0];
        UITableViewCell *cell = [tableView cellForRowAtIndexPath:ip];
        if (i == indexPath.row)
        {
            cell.accessoryType = UITableViewCellAccessoryCheckmark;
        }
        else
        {
            cell.accessoryType = UITableViewCellAccessoryNone;
        }
    }
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    static NSString *CellIdentifier = @"Cell";
    
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier];
    if (cell == nil) {
        cell = [[[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier] autorelease];
    }
    
    // Configure the cell
    cell.accessoryType = UITableViewCellAccessoryNone;
    cell.textLabel.text = [[self.projects objectAtIndex:indexPath.row] titleOfProject];
    
    return cell;
}

- (void)tableView:(UITableView *)tableView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowAtIndexPath:(NSIndexPath *)indexPath
{
    if (editingStyle == UITableViewCellEditingStyleDelete) 
    {
        FABProject *selectedProject = [self.projects objectAtIndex:indexPath.row];
        // Update server
        [self deleteProject:selectedProject];
        
        // Update local variable
        [self.projects removeObjectAtIndex:indexPath.row];
        
        // Refresh table view
        [tableView deleteRowsAtIndexPaths:[NSArray arrayWithObject:indexPath] withRowAnimation:UITableViewRowAnimationFade];
    }
}

- (UITableViewCellEditingStyle)tableView:(UITableView *)tableView editingStyleForRowAtIndexPath:(NSIndexPath *)indexPath
{
    // No editing style if not editing or the index path is nil.
    if (tableView.editing == NO || !indexPath)
    {
        return UITableViewCellEditingStyleNone;
    }
    
    else 
	{
		return UITableViewCellEditingStyleDelete;
	}
    return UITableViewCellEditingStyleNone;
}



@end
