//
//  FABGalleryPhotoViewController.m
//  Fabiji
//
//  Created by Zhao He on 1/31/12.
//  Copyright (c) 2012 Media Computing Group, RWTH Aachen University. All rights reserved.
//


#import "FABGalleryPhotoViewController.h"

@interface FABGalleryPhotoViewController()

typedef enum {
    kLoadProjects,
} FABRequestMethodType;

@property (nonatomic, assign) FABRequestMethodType requestMethod;
@property (nonatomic, retain) NSMutableArray *thumbnailProjects;

@end

@implementation FABGalleryPhotoViewController

@synthesize thumbnailProjects = _thumbnailProjects;
@synthesize requestMethod = _requestMethod;
@synthesize pageControlIsChangingPage = _pageControlIsChangingPage;
@synthesize scrollView = _scroll;
@synthesize pageControl = _pageControl;
@synthesize searchBar = _searchBar;
@synthesize searchController = _searchController;

- (void)dealloc 
{
    [_thumbnailProjects release];
    [_scroll release];
    [_pageControl release];
    [_searchBar release];
    [_searchController release];
    [super dealloc];
}

#pragma mark - RestKit


- (void)loadProjectsAndSetupScrollView
{
    // GET /projects
    [[RKObjectManager sharedManager] loadObjectsAtResourcePath:@"/projects" delegate:self];
    self.requestMethod = kLoadProjects;
}

- (void)request:(RKRequest*)request didLoadResponse:(RKResponse*)response 
{
    NSLog(@"Response: %@", [response bodyAsString]);
    NSLog(@"Status: %d", [response statusCode]);
    if ([response isOK]) {
        NSLog(@"response isOK.");
    }
    else if([response isCreated]) {
        NSLog(@"response isCreated.");
    }
}

- (void)objectLoader:(RKObjectLoader*)objectLoader didLoadObjects:(NSArray *)objects 
{
    NSLog(@"objects: %@", objects);
    if (self.requestMethod == kLoadProjects) 
    {
        self.thumbnailProjects = [[[NSMutableArray alloc] init] autorelease];
        // Find projects that have thumbnails (pictures)
        for (FABProject *project in objects) 
        {
            if ( [project.thumbnailFileOfProject length] )
            {
                [self.thumbnailProjects addObject:project];
            }
        }
        [self setupScrollViewsAndPages];
    }
}

- (void)objectLoader:(RKObjectLoader*)objectLoader didFailWithError:(NSError*)error {
	UIAlertView* alert = [[[UIAlertView alloc] initWithTitle:@"Error" message:[error localizedDescription] delegate:nil cancelButtonTitle:@"OK" otherButtonTitles:nil] autorelease];
	[alert show];
	NSLog(@"Hit error: %@", error);
}


#pragma mark - IBActions

- (IBAction)changePage 
{
	// Update the scroll view to the appropriate page
	CGRect frame;
	frame.origin.x = self.scrollView.frame.size.width * self.pageControl.currentPage;
	frame.origin.y = 0;
	frame.size = self.scrollView.frame.size;
	[self.scrollView scrollRectToVisible:frame animated:YES];
	
	// Keep track of when scrolls happen in response to the page control
	// value changing. If we don't do this, a noticeable "flashing" occurs
	// as the the scroll delegate will temporarily switch back the page
	// number.
	self.pageControlIsChangingPage = YES;
    
    NSLog(@"changePage:");
}

#pragma mark - Private methods


- (void)alignSubviews 
{
	// Position all the content views at their respective page positions
	self.scrollView.contentSize = CGSizeMake(self.pageControl.numberOfPages * self.scrollView.bounds.size.width,
										self.scrollView.bounds.size.height);
	NSUInteger i = 0;
	for (UIView *v in self.scrollView.subviews) {
		v.frame = CGRectMake(i * self.scrollView.bounds.size.width, 0,
							 self.scrollView.bounds.size.width, self.scrollView.bounds.size.height);
		i++;
	}
}

- (void)setupSearchControllerWithProjects:(NSArray *)projects
{
    // Extract project names
    
    NSMutableArray *contents = [[[NSMutableArray alloc] init] autorelease];
    for (FABProject *project in projects) 
    {
        [contents addObject:project.titleOfProject];
    }
    
    // Assign project names to search contents
    self.searchController.contents = contents;
}

- (void)setupPojectThumbnailDelegate
{
    UINavigationController *navigationController = [self.navigationController.tabBarController.viewControllers objectAtIndex:3];
    UISplitViewController *splitViewController = [navigationController.viewControllers objectAtIndex:0];
    FABProjectsDetailViewController *detailViewController = [splitViewController.viewControllers lastObject];
    detailViewController.delegate = self;
}


- (void)setupScrollViewsAndPages
{    
    float numberOfPagesInFloat = [self.thumbnailProjects count] / 6.0;
    int numberOfPages = (int)ceil(numberOfPagesInFloat);
    self.pageControl.numberOfPages = numberOfPages;
    
    // Remove old subviews;
    for (UIView *view in self.scrollView.subviews) 
    {
        [view removeFromSuperview];
        
        // Release view controller now
        [[view viewController] release];
    }
    
    // Configure new views into pages
    int numberOfImagesLeft = [self.thumbnailProjects count];
    
    NSArray *thumbnailProjectsInCurrentPage = nil;
    
    for (int i = 0; i < numberOfPages; i++) 
    {
        if (numberOfImagesLeft > 6) {
            thumbnailProjectsInCurrentPage = [self.thumbnailProjects subarrayWithRange:NSMakeRange(6*i, 6)];
        }
        
        else {
            thumbnailProjectsInCurrentPage = [self.thumbnailProjects subarrayWithRange:NSMakeRange(6*i, numberOfImagesLeft)];
        }
        
        numberOfImagesLeft -= 6;
        
        FABGallerySinglePageViewController *singlePageViewController = [[FABGallerySinglePageViewController alloc] init];
        [self.scrollView addSubview:singlePageViewController.view];
        singlePageViewController.projects = thumbnailProjectsInCurrentPage;
        
        // DO NOT RELEASE HERE! UIButtons are in use
        //[singlePageViewController release];
    }
    
    // Setup search controller
    if (thumbnailProjectsInCurrentPage) {
        [self setupSearchControllerWithProjects:thumbnailProjectsInCurrentPage];
    }
    
    
    // Layout new views into pages
    [self alignSubviews];
}

- (void)willRotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation
								duration:(NSTimeInterval)duration {
    [self.pageControl setCurrentPageBypass:self.scrollView.contentOffset.x / self.scrollView.bounds.size.width];
}

- (void)willAnimateRotationToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
										 duration:(NSTimeInterval)duration {
	[self alignSubviews];
	self.scrollView.contentOffset = CGPointMake(self.pageControl.currentPage * self.scrollView.bounds.size.width, 0);
}


#pragma mark - FABProjectThumbnailDelegate

- (void)thumbnailDidChange
{
    [self loadProjectsAndSetupScrollView];
}


#pragma mark - Scroll view delegate

- (void)scrollViewDidScroll:(UIScrollView *)scrollView
{
    if (self.pageControlIsChangingPage) {
        return;
    }
    
	// We switch page at 50% across
    CGFloat pageWidth = scrollView.frame.size.width;
    int page = floor((scrollView.contentOffset.x - pageWidth / 2) / pageWidth) + 1;
    //self.pageControl.currentPage = page;
    [self.pageControl setCurrentPageBypass:page];
}

- (void)scrollViewDidEndDecelerating:(UIScrollView *)scrollView 
{
    self.pageControlIsChangingPage = NO;
}


#pragma mark - View lifecycle

- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    [self.navigationController setNavigationBarHidden:YES animated:animated];
}

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    self.searchController = [[[FABSearchController alloc]
                              initWithSearchBar:self.searchBar
                              contentsController:self] autorelease];
    
    [self setupPojectThumbnailDelegate];
    
    [self loadProjectsAndSetupScrollView];
}

- (void)viewDidUnload
{
    [self setSearchBar:nil];
    [self setSearchController:nil];
    [super viewDidUnload];
    // Release any retained subviews of the main view.
    // e.g. self.myOutlet = nil;
}


- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	return YES;
}

@end
