//
//  CalendarMasterViewController.m
//  Fabiji
//
//  Created by Zhao He on 19/01/12.
//  Copyright (c) 2012 Media Computing Group, RWTH Aachen University. All rights reserved.
//

#import "FABCalendarMasterViewController.h"

// in the Gregorian calendar, 1 = Sun, 2 = Mon, 3 = Tue
const int TUESDAY = 3;
static NSString* const FabLabCalendarName = @"fablab Aachen";
static NSString* const FabLabMasterURL = @"mailto:rene.bohne@gmail.com";

@implementation FABCalendarMasterViewController

@synthesize eventStore = _eventStore;
@synthesize events = _events;
@synthesize appointmentDate = _appointmentDate;
@synthesize gregorian = _gregorian;
@synthesize appointments = _appointments;
@synthesize detailViewController = _detailViewController;
@synthesize headerView = _headerView;
@synthesize appointmentDateLabel = _appointmentDateLable;

- (void)dealloc
{
    [_eventStore release];
    [_events release];
    [_appointmentDate release];
    [_gregorian release];
    [_appointments release];
    [_detailViewController release];
    [_headerView release];
    [_appointmentDateLable release];
    [super dealloc];
}

#pragma mark - IBActions

- (IBAction)reloadEvents:(id)sender 
{    
    // Find fab lab aachen calendar within Calendar App in the iPad
    EKCalendar *fablabCalendar = nil;
    NSArray *calendars = [self.eventStore calendars];
    for (EKCalendar *cal in calendars) 
    {
        if ([cal.title isEqualToString:FabLabCalendarName]) {
            fablabCalendar = cal;
            break;
        }
    }
    
    if (fablabCalendar == nil) {
        NSLog(@"%@", @"Calendar not found: fablab Aachen");
        return;
    }
    
    // Define a range of event dates we want to display
     NSDateComponents *comps = [[NSDateComponents alloc] init];
    [comps setSecond:60*60*24-1];
    NSDate *startDate = self.appointmentDate;                                                               // start at 00:00:00 on that day
    NSDate *endDate = [self.gregorian dateByAddingComponents:comps toDate:self.appointmentDate options:0];  // end   at 23:59:59 on that day
    [comps release];
    
    // Create a predicate to search all celndars with our date range
    NSPredicate *predicate = [self.eventStore predicateForEventsWithStartDate:startDate endDate:endDate calendars:[NSArray arrayWithObject:fablabCalendar]];
    
    // Query the event store using the predicate.
    NSArray *results       = [self.eventStore eventsMatchingPredicate:predicate];
    
    // Convert the results to a mutable array and store so we can implement swipe to delete
    NSMutableArray *events = [[NSMutableArray alloc] initWithArray:results];
    self.events            = events;
    [events release];
    
    // Configure the table view cell contents - Appointments
    NSMutableArray *newAppointments = [[NSMutableArray alloc] init];
    for (EKEvent *event in self.events) 
    {
        FABAppointment *appointment = [[FABAppointment alloc] init];
        appointment.title = event.title;
        appointment.attendeeName = @"";
        appointment.notes = event.notes;
        appointment.startDate = event.startDate;
        appointment.endDate = event.endDate;
        if ([event hasAttendees]) 
        {
            NSString *attendeeName = @"";
            for (EKParticipant *att in event.attendees) 
            {
                if (att.participantRole != EKParticipantRoleUnknown && ![att.URL.absoluteString isEqualToString:FabLabMasterURL]) {
                    attendeeName = att.name;
                    break;
                }
            }            
            appointment.attendeeName = attendeeName;
        }
        [newAppointments addObject:appointment];
        [appointment release];
    }
    self.appointments = newAppointments;
    [newAppointments release];
    
    
    // Update the table view
    [self.tableView reloadData];
}

- (IBAction)segmentAction:(id)sender
{
	UISegmentedControl *segmentedControl = (UISegmentedControl *)sender;
	int selection = segmentedControl.selectedSegmentIndex;
   
    NSDateComponents *comps = [[NSDateComponents alloc] init];
    switch (selection) {
        case 0:
            [comps setDay:-7];
            break;
        case 1:
            [comps setDay:7];
            break;
        default:
            break;
    }
    
    self.appointmentDate = [self.gregorian dateByAddingComponents:comps toDate:self.appointmentDate options:0];
    [self setAppointmentDateLabel];
    [self reloadEvents:nil];

    [comps release];    
}

// In fact it is this week's events
- (IBAction)todayEvents:(id)sender
{
    NSDate *now = [NSDate date];
    NSDate *nextTuesday;
    NSDateComponents *weekdayComponents =[self.gregorian components:NSWeekdayCalendarUnit fromDate:now];
    NSInteger weekday = [weekdayComponents weekday];
    
    // find beginning of date
    NSDate *beginingOfNow = [self beginningOfDay:now];
    
    // Find the next Tuesday if today is not Tuesday
    if (weekday == TUESDAY) {
        nextTuesday = beginingOfNow;
    }
    else 
    {
        int daysToAdd = TUESDAY - weekday;
        if (daysToAdd < 0) {
            daysToAdd += 7;
        }
        NSDateComponents *addComponents = [[NSDateComponents alloc] init];
        [addComponents setDay:daysToAdd];
        nextTuesday = [self.gregorian dateByAddingComponents:addComponents toDate:beginingOfNow options:0];
        [addComponents release];
    }
    
    self.appointmentDate = nextTuesday;
    [self setAppointmentDateLabel];
    [self reloadEvents:nil];
}


#pragma mark - Private methods

// Return the date at 00:00:00
- (NSDate *)beginningOfDay:(NSDate*)date
{
    unsigned int flags = NSYearCalendarUnit | NSMonthCalendarUnit | NSDayCalendarUnit;
    NSDateComponents* components = [self.gregorian components:flags fromDate:date];
    NSDate* dateOnly = [self.gregorian dateFromComponents:components];
    
    return dateOnly;
}


- (void)setAppointmentDateLabel
{
    // Reformat string
    NSDateFormatter *dateFormat = [[NSDateFormatter alloc] init];
    [dateFormat setDateFormat:@"EEE MMMM d, YYYY"];
    NSString *appointmentDateString = [dateFormat stringFromDate:self.appointmentDate];
    [dateFormat release];
    
    self.appointmentDateLabel.text = appointmentDateString;
}

#pragma mark - View lifecycle

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    // Initialize variables
    _eventStore = [[EKEventStore alloc] init];
    _gregorian = [[NSCalendar alloc] initWithCalendarIdentifier:NSGregorianCalendar];
    _appointments = [[NSMutableArray alloc] init];
    
    // set up the table's header view based on 'headerView' outlet
	CGRect newFrame = CGRectMake(0.0, 0.0, self.tableView.bounds.size.width, self.headerView.frame.size.height);
	self.headerView.backgroundColor = [UIColor clearColor];
	self.headerView.frame = newFrame;
	self.tableView.tableHeaderView = self.headerView;
    
    // Add segmented control to the top center
	NSArray *segmentTextContent = [NSArray arrayWithObjects:@"Last", @"Next", nil];
	UISegmentedControl *segmentedControl = [[UISegmentedControl alloc] initWithItems:segmentTextContent];
	//segmentedControl.selectedSegmentIndex = 0;
    segmentedControl.frame = CGRectMake(0, 0, 160, 30);
	segmentedControl.segmentedControlStyle = UISegmentedControlStyleBar;
    segmentedControl.momentary = YES;
	[segmentedControl addTarget:self action:@selector(segmentAction:) forControlEvents:UIControlEventValueChanged];
    
	self.navigationItem.titleView = segmentedControl;
	[segmentedControl release];
    
    
    // Add a refresh button to the top right
    UIBarButtonItem *refreshButton = [[UIBarButtonItem alloc] initWithBarButtonSystemItem:UIBarButtonSystemItemRefresh target:self action:@selector(reloadEvents:)];
    self.navigationItem.rightBarButtonItem = refreshButton;
    [refreshButton release];
    
    
    // Add a today button to the top left
    UIBarButtonItem *todayButton = [[UIBarButtonItem alloc] initWithTitle:@"Today" style:UIBarButtonItemStylePlain target:self action:@selector(todayEvents:)];
    self.navigationItem.leftBarButtonItem = todayButton;
    [todayButton release];
    
    
    // Load events
    [self todayEvents:nil];
}

- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    return YES;
}

#pragma mark - Table view

// Customize the number of sections in the table view
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return 1;
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return [self.appointments count];
}

// Customize the appearance of table view cells
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    static NSString *CellIdentifier = @"Cell";
    
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier];
    if (cell == nil) {
        cell = [[[UITableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle reuseIdentifier:CellIdentifier] autorelease];
    }

    cell.textLabel.text = [[self.appointments objectAtIndex:indexPath.row] title];
    cell.detailTextLabel.text = [[self.appointments objectAtIndex:indexPath.row] attendeeName];
    cell.accessoryType = UITableViewCellAccessoryNone;
        
    return cell;
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    //[tableView deselectRowAtIndexPath:indexPath animated:YES];
    FABAppointment *detailItem = [self.appointments objectAtIndex:indexPath.row];
    if (self.detailViewController) {
        self.detailViewController.detailItem = detailItem;
    }
}

- (NSString *)tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section {
    return @"Appointments";
}


@end
