//
//  GestureCaptureAppDelegate.m
//  GestureCapture
//
//  Created by D.Stengele on 22/05/2011.
//

#import "GestureCaptureAppDelegate.h"

#define PRINT_FIELDS          1
#define SEARCH_UNDOC_FIELDS   1
#define PRINT_UNDOC_AS_DOUBLE 0
#define PRINT_BUFFER          1
#define WRITE_BUFFER_TO_FILE  0
#define MANIP_EVENT           1    // Copy event, manipulate copy and compare byte buffer with original
#define TOUCHES_CONDITION     >=1  // Disassemble touches only, if the count of touches satisfies this condition



@implementation GestureCaptureAppDelegate

static FILE* file = NULL;

@synthesize window;

- (void)applicationDidFinishLaunching:(NSNotification*)aNotification
{
    eventTap = NULL;
    runLoopSource = NULL;
    file = NULL;
    [lbStatus setStringValue:@"Off"];

    #if WRITE_BUFFER_TO_FILE
    file = fopen("/TouchRecords.txt", "w");
    if(!file)
        NSLog(@"Failed to open file");
    else
    {
        fprintf(file, "Touch\t");
        fprintf(file, "X\t");
        fprintf(file, "Y\t");
        fprintf(file, "n\t");
        for(int i = 0; i < 468; ++i)
            fprintf(file, "#%03d\t", i);
        fprintf(file, "\n");
        fflush(file);
    }
    #endif
}

- (void)applicationWillTerminate:(NSNotification*)notification
{
    if(eventTap)
    {
        CGEventTapEnable(eventTap, false);
        CFRunLoopRemoveSource(CFRunLoopGetCurrent(), runLoopSource, kCFRunLoopCommonModes);
        CFRelease((CFTypeRef)eventTap);
        CFRelease((CFTypeRef)runLoopSource);
        eventTap = NULL;
        runLoopSource = NULL;
    }
    if(file)
    {
        fflush(file);
        fclose(file);
        file = NULL;
    }
}

- (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication*)sender
{
    return YES;
}



// Universal Access is required to use Keyboard Event Taps. This method returns YES if they are enabeld.
// If they're not enabled, the user needs to enable "Enable access for assistive devices" in the
//   "Universal Access" section in the System Preferences.
- (BOOL)isUniversalAccessEnabled
{
    return AXAPIEnabled() || AXIsProcessTrusted();
}

void writeBufferToFile(uint8* buffer, uint n, int touchState, CGPoint touchPoint)
{
    if(file)
    { 
        fprintf(file, "%d\t", touchState);
        fprintf(file, "%1.2f\t", touchPoint.x);
        fprintf(file, "%1.2f\t", touchPoint.y);
        fprintf(file, "%u\t", n);

        for(uint i = 0; i < n; ++i)
            fprintf(file, "%03u\t", buffer[i]);

        fprintf(file, "\n");
        fflush(file);
    }
    else
        NSLog(@"Can't write to file, because it's not opened.");
}

const char* eventTypeToNSEventString(CGEventType type)
{
    switch(type)
    {
        case NSLeftMouseDown:
            return "NSLeftMouseDown";

        case NSLeftMouseUp:
            return "NSLeftMouseUp";

        case NSRightMouseDown:
            return "NSRightMouseDown";

        case NSRightMouseUp:
            return "NSRightMouseUp";

        case NSOtherMouseDown:
            return "NSOtherMouseDown";

        case NSOtherMouseUp:
            return "NSOtherMouseUp";

        case NSMouseMoved:
            return "NSMouseMoved";

        case NSLeftMouseDragged:
            return "NSLeftMouseDragged";

        case NSRightMouseDragged:
            return "NSRightMouseDragged";

        case NSOtherMouseDragged:
            return "NSOtherMouseDragged";

        case NSMouseEntered:
            return "NSMouseEntered";

        case NSMouseExited:
            return "NSMouseExited";

        case NSCursorUpdate:
            return "NSCursorUpdate";

        case NSKeyDown:
            return "NSKeyDown";

        case NSKeyUp:
            return "NSKeyUp";

        case NSFlagsChanged:
            return "NSFlagsChanged";

        case NSAppKitDefined:
            return "NSAppKitDefined";

        case NSSystemDefined:
            return "NSSystemDefined";

        case NSApplicationDefined:
            return "NSApplicationDefined";

        case NSPeriodic:
            return "NSPeriodic";

        case NSScrollWheel:
            return "NSScrollWheel";

        case NSTabletPoint:
            return "NSTabletPoint";

        case NSTabletProximity:
            return "NSTabletProximity";

        case NSEventTypeGesture:
            return "NSEventTypeGesture";

        case NSEventTypeMagnify:
            return "NSEventTypeMagnify";

        case NSEventTypeSwipe:
            return "NSEventTypeSwipe";

        case NSEventTypeRotate:
            return "NSEventTypeRotate";

        case NSEventTypeBeginGesture:
            return "NSEventTypeBeginGesture";

        case NSEventTypeEndGesture:
            return "NSEventTypeEndGesture";

        default:
            return "Unknown NSEvent";
    }
}

const char* eventTypeToCGEventString(CGEventType type)
{
    switch(type)
    {
        case kCGEventNull:
            return "kCGEventNull";

        case kCGEventLeftMouseDown:
            return "kCGEventLeftMouseDown";

        case kCGEventLeftMouseUp:
            return "kCGEventLeftMouseUp";

        case kCGEventRightMouseDown:
            return "kCGEventRightMouseDown";

        case kCGEventRightMouseUp:
            return "kCGEventRightMouseUp";

        case kCGEventMouseMoved:
            return "kCGEventMouseMoved";

        case kCGEventLeftMouseDragged:
            return "kCGEventLeftMouseDragged";

        case kCGEventRightMouseDragged:
            return "kCGEventRightMouseDragged";

        case kCGEventKeyDown:
            return "kCGEventKeyDown";

        case kCGEventKeyUp:
            return "kCGEventKeyUp";

        case kCGEventFlagsChanged:
            return "kCGEventFlagsChanged";

        case kCGEventScrollWheel:
            return "kCGEventScrollWheel";

        case kCGEventTabletPointer:
            return "kCGEventTabletPointer";

        case kCGEventTabletProximity:
            return "kCGEventTabletProximity";

        case kCGEventOtherMouseDown:
            return "kCGEventOtherMouseDown";

        case kCGEventOtherMouseUp:
            return "kCGEventOtherMouseUp";

        case kCGEventOtherMouseDragged:
            return "kCGEventOtherMouseDragged";

        case kCGEventTapDisabledByTimeout:
            return "kCGEventTapDisabledByTimeout";

        case kCGEventTapDisabledByUserInput:
            return "kCGEventTapDisabledByUserInput";

        default:
            return "Unknown CGEvent";
    }
}

const char* eventFlagToString(CGEventFlags flag)
{
    switch(flag)
    {
        case kCGEventFlagMaskAlphaShift:
            return "kCGEventFlagMaskAlphaShift";
            
        case kCGEventFlagMaskShift:
            return "kCGEventFlagMaskShift";
            
        case kCGEventFlagMaskControl:
            return "kCGEventFlagMaskControl";
            
        case kCGEventFlagMaskAlternate:
            return "kCGEventFlagMaskAlternate";
            
        case kCGEventFlagMaskCommand:
            return "kCGEventFlagMaskCommand";
            
        case kCGEventFlagMaskHelp:
            return "kCGEventFlagMaskHelp";
            
        case kCGEventFlagMaskSecondaryFn:
            return "kCGEventFlagMaskSecondaryFn";
            
        case kCGEventFlagMaskNumericPad:
            return "kCGEventFlagMaskNumericPad";
            
        case kCGEventFlagMaskNonCoalesced:
            return "kCGEventFlagMaskNonCoalesced";
            
        default:
            return "Unknown Event Flag";
    }
}

const char* eventFieldToString(CGEventField field)
{
    switch(field)
    {
        case kCGMouseEventNumber:
            return "kCGMouseEventNumber";
            
        case kCGMouseEventClickState:
            return "kCGMouseEventClickState";
            
        case kCGMouseEventPressure:
            return "kCGMouseEventPressure";
            
        case kCGMouseEventButtonNumber:
            return "kCGMouseEventButtonNumber";
            
        case kCGMouseEventDeltaX:
            return "kCGMouseEventDeltaX";
            
        case kCGMouseEventDeltaY:
            return "kCGMouseEventDeltaY";
            
        case kCGMouseEventInstantMouser:
            return "kCGMouseEventInstantMouser";
            
        case kCGMouseEventSubtype:
            return "kCGMouseEventSubtype";
            
        case kCGKeyboardEventAutorepeat:
            return "kCGKeyboardEventAutorepeat";
            
        case kCGKeyboardEventKeycode:
            return "kCGKeyboardEventKeycode";
            
        case kCGKeyboardEventKeyboardType:
            return "kCGKeyboardEventKeyboardType";
            
        case kCGScrollWheelEventDeltaAxis1:
            return "kCGScrollWheelEventDeltaAxis1";
            
        case kCGScrollWheelEventDeltaAxis2:
            return "kCGScrollWheelEventDeltaAxis2";
            
        case kCGScrollWheelEventDeltaAxis3:
            return "kCGScrollWheelEventDeltaAxis3";
            
        case kCGScrollWheelEventFixedPtDeltaAxis1:
            return "kCGScrollWheelEventFixedPtDeltaAxis1";
            
        case kCGScrollWheelEventFixedPtDeltaAxis2:
            return "kCGScrollWheelEventFixedPtDeltaAxis2";
            
        case kCGScrollWheelEventFixedPtDeltaAxis3:
            return "kCGScrollWheelEventFixedPtDeltaAxis3";
            
        case kCGScrollWheelEventPointDeltaAxis1:
            return "kCGScrollWheelEventPointDeltaAxis1";
            
        case kCGScrollWheelEventPointDeltaAxis2:
            return "kCGScrollWheelEventPointDeltaAxis2";
            
        case kCGScrollWheelEventPointDeltaAxis3:
            return "kCGScrollWheelEventPointDeltaAxis3";
            
        case kCGScrollWheelEventInstantMouser:
            return "kCGScrollWheelEventInstantMouser";
            
        case kCGTabletEventPointX:
            return "kCGTabletEventPointX";
            
        case kCGTabletEventPointY:
            return "kCGTabletEventPointY";
            
        case kCGTabletEventPointZ:
            return "kCGTabletEventPointZ";
            
        case kCGTabletEventPointButtons:
            return "kCGTabletEventPointButtons";
            
        case kCGTabletEventPointPressure:
            return "kCGTabletEventPointPressure";
            
        case kCGTabletEventTiltX:
            return "kCGTabletEventTiltX";
            
        case kCGTabletEventTiltY:
            return "kCGTabletEventTiltY";
            
        case kCGTabletEventRotation:
            return "kCGTabletEventRotation";
            
        case kCGTabletEventTangentialPressure:
            return "kCGTabletEventTangentialPressure";
            
        case kCGTabletEventDeviceID:
            return "kCGTabletEventDeviceID";
            
        case kCGTabletEventVendor1:
            return "kCGTabletEventVendor1";
            
        case kCGTabletEventVendor2:
            return "kCGTabletEventVendor2";
            
        case kCGTabletEventVendor3:
            return "kCGTabletEventVendor3";
            
        case kCGTabletProximityEventVendorID:
            return "kCGTabletProximityEventVendorID";
            
        case kCGTabletProximityEventTabletID:
            return "kCGTabletProximityEventTabletID";
            
        case kCGTabletProximityEventPointerID:
            return "kCGTabletProximityEventPointerID";
            
        case kCGTabletProximityEventDeviceID:
            return "kCGTabletProximityEventDeviceID";
            
        case kCGTabletProximityEventSystemTabletID:
            return "kCGTabletProximityEventSystemTabletID";
            
        case kCGTabletProximityEventVendorPointerType:
            return "kCGTabletProximityEventVendorPointerType";
            
        case kCGTabletProximityEventVendorPointerSerialNumber:
            return "kCGTabletProximityEventVendorPointerSerialNumber";
            
        case kCGTabletProximityEventVendorUniqueID:
            return "kCGTabletProximityEventVendorUniqueID";
            
        case kCGTabletProximityEventCapabilityMask:
            return "kCGTabletProximityEventCapabilityMask";
            
        case kCGTabletProximityEventPointerType:
            return "kCGTabletProximityEventPointerType";
            
        case kCGTabletProximityEventEnterProximity:
            return "kCGTabletProximityEventEnterProximity";
            
        case kCGEventTargetProcessSerialNumber:
            return "kCGEventTargetProcessSerialNumber";
            
        case kCGEventTargetUnixProcessID:
            return "kCGEventTargetUnixProcessID";
            
        case kCGEventSourceUnixProcessID:
            return "kCGEventSourceUnixProcessID";
            
        case kCGEventSourceUserData:
            return "kCGEventSourceUserData";
            
        case kCGEventSourceUserID:
            return "kCGEventSourceUserID";
            
        case kCGEventSourceGroupID:
            return "kCGEventSourceGroupID";
            
        case kCGEventSourceStateID:
            return "kCGEventSourceStateID";
            
        case kCGScrollWheelEventIsContinuous:
            return "kCGScrollWheelEventIsContinuous";

        default:
            return "Unknown Event Field";
    }
}

const char* touchPhaseToString(NSTouchPhase touchPhase)
{
    switch(touchPhase)
    {
        case NSTouchPhaseBegan:
            return "NSTouchPhaseBegan";

        case NSTouchPhaseMoved:
            return "NSTouchPhaseMoved";

        case NSTouchPhaseStationary:
            return "NSTouchPhaseStationary";

        case NSTouchPhaseEnded:
            return "NSTouchPhaseEnded";

        case NSTouchPhaseCancelled:
            return "NSTouchPhaseCancelled";

        case NSTouchPhaseTouching:
            return "NSTouchPhaseTouching";

        case NSTouchPhaseAny:
            return "NSTouchPhaseAny";

        default:
            return "Unknown Touch Phase";
    }
}

NSString* stringWithTouchInformationFromEvent(NSEvent* event)
{
    NSSet* touches = [event touchesMatchingPhase:NSTouchPhaseAny inView:nil];
    NSString* ret = [NSString stringWithFormat:@"Touches(%u)=\n   {", [touches count]];
    if([touches count])
    {
        NSEnumerator* e = [touches objectEnumerator];
        NSTouch* touch = nil;
        while((touch = [e nextObject]))
            ret = [ret stringByAppendingFormat:@"\n      (%f/%f), %s", touchPhaseToString(touch.phase), touch.normalizedPosition.x, touch.normalizedPosition.y];
    }
    else
        ret = [ret stringByAppendingString:@"\n      (None)"];
    return ret = [ret stringByAppendingString:@"\n   }"];
}

void printBuffer(UInt8* buffer, UInt16 n)
{
    printf("\nBuffer: %d byte\n", n);
    int i = 0;
    for(int row = 0; i < n; ++row)
    {
        printf("[%03d]:", 10*row);
        for(int column = 0; column < 10 && i < n; ++column)
        {
            if(buffer[i])
                printf(" %03u", buffer[i]);
            else
                printf(" ---");
            ++i;
        }
        printf("\n");
    }
}

void printBufferHex(UInt8* buffer, UInt16 n)
{
    printf("\nBuffer: %d byte (Values: Hex)\n", n);
    int i = 0;
    for(int row = 0; i < n; ++row)
    {
        printf("[%03d]:", 10*row);
        for(int column = 0; column < 10 && i < n; ++column)
        {
            if(buffer[i])
                printf(" %02X", buffer[i]);
            else
                printf(" --");
            ++i;
        }
        printf("\n");
    }
}

void compareBuffer(UInt8* buffer1, UInt8* buffer2, UInt16 n)
{
    for(UInt16 i = 0U; i < n; ++i)
        if(buffer1[i] != buffer2[i])
            NSLog(@"buffer1[%03u] = %03d, buffer2[%03u] = %03d", i, buffer1[i], i, buffer2[i]);
}

void compareBufferHex(UInt8* buffer1, UInt8* buffer2, UInt16 n)
{
    for(UInt16 i = 0U; i < n; ++i)
        if(buffer1[i] != buffer2[i])
            NSLog(@"buffer1[%03u] = 0x%02X, buffer2[%03u] = 0x%02X", i, buffer1[i], i, buffer2[i]);
}

UInt8* createBufferFromEvent(CGEventRef event, UInt16* n)
{
    CFDataRef dataRef = CGEventCreateData(kCFAllocatorDefault, event);
    if(dataRef)
    {
        *n = (UInt16)CFDataGetLength(dataRef);
        UInt8* buffer = (UInt8*)malloc(sizeof(UInt8)*(*n));
        if(buffer)
            CFDataGetBytes(dataRef, CFRangeMake(0, (CFIndex)*n), buffer);
        CFRelease(dataRef);
        return buffer;
    }
    return NULL;
}

// Returns count of occurences of "value" (with size "valueSize") in "buffer" (with size "n")
UInt16 searchValueInBuffer(UInt8* buffer, UInt16 n, UInt8* value, UInt16 valueSize)
{
    UInt16 foundCount = 0U;

    // Copy value and swap bytes
    UInt8* valueSwapped = (UInt8*)malloc(valueSize);
    if(!valueSwapped)
        return UINT16_MAX;
    for(UInt16 i = 0U; i < valueSize; ++i)
        valueSwapped[i] = value[valueSize-i-1U];

    // Search value
    for(UInt16 i = 0U; i < n-valueSize; ++i)
        if(!memcmp(valueSwapped, &buffer[i], valueSize))
            NSLog(@"MATCH #%u: Value at buffer[%03u-%03u]", ++foundCount, i, i+valueSize-1U);

    free(valueSwapped);
    valueSwapped = NULL;

    return foundCount;
}

NSString* descriptionForCGEvent(CGEventRef event)
{
    // Basic values
    NSString* basic = [NSString stringWithFormat:@"CGEvent (0x%X)\n(\n   type=%d, typeID=%d, location=(%f/%f), timestamp=%llu",
                                                 (int)event,
                                                 (int)CGEventGetType(event),
                                                 (int)CGEventGetTypeID(),
                                                 CGEventGetLocation(event).x, CGEventGetLocation(event).y, // Relative to upper-left corner of the main display
                                                 CGEventGetTimestamp(event)];

    // Flags
    NSString* flags = @"flags={";
    static CGEventFlags allFlags[] =
    {
        kCGEventFlagMaskAlphaShift,   // Caps Lock key is down for a keyboard, mouse, or flag-changed event
        kCGEventFlagMaskShift,        // Shift key is down for a keyboard, mouse, or flag-changed event
        kCGEventFlagMaskControl,      // Control key is down for a keyboard, mouse, or flag-changed event
        kCGEventFlagMaskAlternate,    // Alt or Option key is down for a keyboard, mouse, or flag-changed event
        kCGEventFlagMaskCommand,      // Command key is down for a keyboard, mouse, or flag-changed event
        kCGEventFlagMaskHelp,         // Help modifier key is down for a keyboard, mouse, or flag-changed event (Not present on most keyboards, Different than the Help key found in the same row as Home and Page Up)
        kCGEventFlagMaskSecondaryFn,  // Fn (Function) key is down for a keyboard, mouse, or flag-changed event (Found primarily on laptop keyboards)
        kCGEventFlagMaskNumericPad,   // Key events from the numeric keypad area on extended keyboards
        kCGEventFlagMaskNonCoalesced, // Mouse and pen movement events are not being coalesced
        0 // Indicates end of array
    };
    CGEventFlags eventFlags = CGEventGetFlags(event);
    if(eventFlags)
    {
        BOOL firstFlag = YES; // For setting the comma the right way
        int i = 0;
        CGEventFlags currentEventFlag = allFlags[i];
        while(currentEventFlag)
        {
            if(currentEventFlag&eventFlags)
                if(firstFlag)
                {
                    flags = [flags stringByAppendingFormat:@"%s", eventFlagToString(currentEventFlag)];
                    firstFlag = NO;
                }
                else
                    flags = [flags stringByAppendingFormat:@", %s", eventFlagToString(currentEventFlag)];
            currentEventFlag = allFlags[++i];
        }
    }
    flags = [flags stringByAppendingString:@"}"];

    // Fields
    NSString* fields = @"fields=\n   {";
    #define VALUE_INT     (int)0
    #define VALUE_DOUBLE  (int)1
    #define VALUE_INVALID (int)-1 // end of array
    struct EventFieldWithType
    {
        CGEventField field;
        int valueType; // as specified through the constants above
    } allFields[] =
    {
        // See "Quartz Event Services Reference" for detailed information
        {kCGMouseEventNumber,                              VALUE_INT   },
        {kCGMouseEventClickState,                          VALUE_INT   },
        {kCGMouseEventPressure,                            VALUE_DOUBLE},
        {kCGMouseEventButtonNumber,                        VALUE_INT   },
        {kCGMouseEventDeltaX,                              VALUE_INT   },
        {kCGMouseEventDeltaY,                              VALUE_INT   },
        {kCGMouseEventInstantMouser,                       VALUE_INT   },
        {kCGMouseEventSubtype,                             VALUE_INT   },
        {kCGKeyboardEventAutorepeat,                       VALUE_INT   },
        {kCGKeyboardEventKeycode,                          VALUE_INT   },
        {kCGKeyboardEventKeyboardType,                     VALUE_INT   },
        {kCGScrollWheelEventDeltaAxis1,                    VALUE_INT   },
        {kCGScrollWheelEventDeltaAxis2,                    VALUE_INT   },
        //{kCGScrollWheelEventDeltaAxis3,                    VALUE_INT   }, // not used (see documentation)
        {kCGScrollWheelEventFixedPtDeltaAxis1,             VALUE_DOUBLE},
        {kCGScrollWheelEventFixedPtDeltaAxis2,             VALUE_DOUBLE},
        //{kCGScrollWheelEventFixedPtDeltaAxis3,             VALUE_DOUBLE},
        {kCGScrollWheelEventPointDeltaAxis1,               VALUE_INT   },
        {kCGScrollWheelEventPointDeltaAxis2,               VALUE_INT   },
        //{kCGScrollWheelEventPointDeltaAxis3,               VALUE_INT   },
        {kCGScrollWheelEventInstantMouser,                 VALUE_INT   },
        {kCGTabletEventPointX,                             VALUE_INT   },
        {kCGTabletEventPointY,                             VALUE_INT   },
        {kCGTabletEventPointZ,                             VALUE_INT   },
        {kCGTabletEventPointButtons,                       VALUE_INT   },
        {kCGTabletEventPointPressure,                      VALUE_DOUBLE},
        {kCGTabletEventTiltX,                              VALUE_DOUBLE},
        {kCGTabletEventTiltY,                              VALUE_DOUBLE},
        {kCGTabletEventRotation,                           VALUE_DOUBLE},
        {kCGTabletEventTangentialPressure,                 VALUE_DOUBLE},
        {kCGTabletEventDeviceID,                           VALUE_INT   },
        {kCGTabletEventVendor1,                            VALUE_INT   },
        {kCGTabletEventVendor2,                            VALUE_INT   },
        {kCGTabletEventVendor3,                            VALUE_INT   },
        {kCGTabletProximityEventVendorID,                  VALUE_INT   },
        {kCGTabletProximityEventTabletID,                  VALUE_INT   },
        {kCGTabletProximityEventPointerID,                 VALUE_INT   },
        {kCGTabletProximityEventDeviceID,                  VALUE_INT   },
        {kCGTabletProximityEventSystemTabletID,            VALUE_INT   },
        {kCGTabletProximityEventVendorPointerType,         VALUE_INT   },
        {kCGTabletProximityEventVendorPointerSerialNumber, VALUE_INT   },
        {kCGTabletProximityEventVendorUniqueID,            VALUE_INT   },
        {kCGTabletProximityEventCapabilityMask,            VALUE_INT   },
        {kCGTabletProximityEventPointerType,               VALUE_INT   },
        {kCGTabletProximityEventEnterProximity,            VALUE_INT   },
        {kCGEventTargetProcessSerialNumber,                VALUE_INT   },
        {kCGEventTargetUnixProcessID,                      VALUE_INT   },
        {kCGEventSourceUnixProcessID,                      VALUE_INT   },
        {kCGEventSourceUserData,                           VALUE_INT   },
        {kCGEventSourceUserID,                             VALUE_INT   },
        {kCGEventSourceGroupID,                            VALUE_INT   },
        {kCGEventSourceStateID,                            VALUE_INT   },
        {kCGScrollWheelEventIsContinuous,                  VALUE_INT   },
        {0U, VALUE_INVALID} // Indicates end of array
    };
    #if PRINT_FIELDS
    int i = 0;
    int intVal = 0;
    double doubleVal = 0.0;
    struct EventFieldWithType currentEventField = allFields[i];
    while(currentEventField.valueType != VALUE_INVALID)
    {
        if(currentEventField.valueType == VALUE_INT)
        {
            intVal = (int)CGEventGetIntegerValueField(event, currentEventField.field);
            if(intVal)
                fields = [fields stringByAppendingFormat:@"\n      %s=%d", eventFieldToString(currentEventField.field), intVal];
        }
        else
        {
            doubleVal = (double)CGEventGetDoubleValueField(event, currentEventField.field);
            if(doubleVal != 0.0)
                fields = [fields stringByAppendingFormat:@"\n      %s=%f", eventFieldToString(currentEventField.field), doubleVal];
        }
        currentEventField = allFields[++i];
    }
    #else
    fields = [fields stringByAppendingString:@"\n      (switched off)"];
    #endif
    fields = [fields stringByAppendingString:@"\n   }"];

    // Touches
    NSEvent* nsEvent = [NSEvent eventWithCGEvent:event];
    NSString* touches = stringWithTouchInformationFromEvent(nsEvent);

    // Disassemble event, if interesting
    if([[nsEvent touchesMatchingPhase:NSTouchPhaseAny inView:nil] count] TOUCHES_CONDITION)
    {
        #if SEARCH_UNDOC_FIELDS
        // Try to find undocumented CGEventField-values
        int j = 1000; // Only found undocumented values <200 until now. So this should be enough.
        do
        {
            // Check if documented
            int k = 0;
            BOOL documented = NO;
            struct EventFieldWithType currentEventField = allFields[k];
            while(!documented && currentEventField.valueType != VALUE_INVALID)
            {
                if(j == (int)currentEventField.field)
                    documented = YES;
                else
                    currentEventField = allFields[++k];
            }

            if(!documented)
            {
                #if PRINT_UNDOC_AS_DOUBLE
                double doubleVal = CGEventGetDoubleValueField(event, (CGEventField)j);
                if(doubleVal != 0.0)
                    fields = [fields stringByAppendingFormat:@"\n   UNDOCUMENTED field %3d = %f", j, doubleVal];
                #else
                int intVal = (int)CGEventGetIntegerValueField(event, (CGEventField)j);
                if(intVal)
                    fields = [fields stringByAppendingFormat:@"\n   UNDOCUMENTED field %3d = %d", j, intVal];
                #endif
            }

            --j;

        }while(j); // No need to check j=0, because this value is documented
        #endif

        // Disassemble event
        UInt16 bufferLength = 0U;
        UInt8* buffer = createBufferFromEvent(event, &bufferLength);
        if(buffer)
        {
            NSTouch* touch = (NSTouch*)[[nsEvent touchesMatchingPhase:NSTouchPhaseAny inView:nil] anyObject];

            #if WRITE_BUFFER_TO_FILE
            writeBufferToFile(buffer, bufferLength, (int)touch.phase, CGPointMake(touch.normalizedPosition.x, touch.normalizedPosition.y));
            #endif

            #if PRINT_BUFFER
            printBufferHex(buffer, bufferLength);
            #endif

            // Find float-value in buffer
            UInt8* value = (UInt8*)malloc(sizeof(Float32));
            Float32* valueF = (Float32*)value;
            *valueF = (Float32)touch.normalizedPosition.y; // Value to search for
            if(!searchValueInBuffer(buffer, bufferLength, value, (UInt16)sizeof(Float32)))
                NSLog(@"NO MATCH ... :-(");
            free(value);
            value = NULL;

            #if MANIP_EVENT
            // Manipulate Event-Copy and compare buffer
            CGEventRef event2 = CGEventCreateCopy(event);
            //CGEventSetLocation(event2, CGPointMake(CGEventGetLocation(event).x, CGEventGetLocation(event).y));
            //CGEventSetLocation(event2, CGPointMake(CGEventGetLocation(event).x+1.0f, CGEventGetLocation(event).y));
            CGEventSetLocation(event2, CGPointMake(0.0f, 0.0f));
            NSLog(@"Event Manipulation enabled (Changed location from (%f/%f) to (%f/%f))", CGEventGetLocation(event).x, CGEventGetLocation(event).y, CGEventGetLocation(event2).x, CGEventGetLocation(event2).y);
            UInt16 buffer2Length = 0U;
            UInt8* buffer2 = createBufferFromEvent(event2, &buffer2Length);
            if(buffer2)
            {
                if(bufferLength != buffer2Length)
                    NSLog(@"Copy has DIFFERENT size! (Original: %d, Copy: %d)", bufferLength, buffer2Length);
                compareBufferHex(buffer, buffer2, MIN(bufferLength, buffer2Length));
                free(buffer2);
                buffer2 = NULL;
            }
            else
                NSLog(@"UNABLE to create Buffer from Event-Copy");
            #endif

            free(buffer);
            buffer = NULL;
        }
        else
            NSLog(@"UNABLE to create Buffer from Event");
    }

    return [NSString stringWithFormat:@"%@\n   %@\n   %@\n   %@\n)\n= = = = = = = = = = = = = = = = = = = =\n", basic, flags, fields, touches];
}

/*
 Perform the following conversions and check if done right:
  1. CGEventRef  => CFDataRef
  2. CFDataRef   => Byte-Buffer
  3. Byte-Buffer => CFDataRef
  4. CFDataRef   => CGEventRef
  5. CGEventRef  => NSEvent*
*/
void convertEvent(CGEventRef event)
{
    // CGEventRef => CFDataRef
    CFDataRef cfDataRef = CGEventCreateData(kCFAllocatorDefault, event);
    if(!cfDataRef)
        NSLog(@"FAILED: CGEventRef => CFDataRef");
    else
    {
        // CFDataRef => Byte-Buffer
        uint l = (uint)CFDataGetLength(cfDataRef);
        UInt8* buffer = (UInt8*)malloc(sizeof(UInt8)*l);
        CFDataGetBytes(cfDataRef, CFRangeMake(0, (CFIndex)l), buffer);
        if(!buffer)
            NSLog(@"FAILED: CFDataRef => Byte-Buffer");
        else
        {
            // Byte-Buffer => CFDataRef
            CFDataRef cfDataRef2 = CFDataCreateWithBytesNoCopy(kCFAllocatorDefault, buffer, (CFIndex)l, kCFAllocatorNull);
            if(!cfDataRef2)
                NSLog(@"FAILED: Byte-Buffer => CFDataRef");
            else
            {
                // CFDataRef => CGEventRef
                CGEventRef cgEventRef = CGEventCreateFromData(kCFAllocatorDefault, cfDataRef2);
                if(!cgEventRef)
                    NSLog(@"FAILED: CFDataRef => CGEventRef");
                else
                {
                    // CGEventRef => NSEvent*
                    NSEvent* nsEvent = [NSEvent eventWithCGEvent:cgEventRef];
                    if(!nsEvent)
                        NSLog(@"FAILED: CGEventRef => NSEvent*");
                    else
                        NSLog(@"Conversion done");
                    CFRelease(cgEventRef);
                    cgEventRef = NULL;
                }
                CFRelease(cfDataRef2);
                cfDataRef2 = NULL;
            }
            free(buffer);
            buffer = NULL;
        }
        CFRelease(cfDataRef);
        cfDataRef = NULL;
    }
}

// Calback function for captured events. Return the event to let it pass (feel free to manipulate it before) or NULL to reject it.
CGEventRef cgEventCallbackFunction(CGEventTapProxy proxy, CGEventType type, CGEventRef event, void* refcon)
{
    (void)proxy;
    (void)type;
    (void)refcon;

    NSLog(@"%@", descriptionForCGEvent(event));

    //convertEvent(event);

    return event;
}

- (IBAction)captureGestures:(id)sender
{
    if(eventTap)
    {
        if(!CGEventTapIsEnabled(eventTap))
        {
            CGEventTapEnable(eventTap, true);
            [lbStatus setStringValue:@"Re-Enabled"];
        }
        return;
    }

    // Note that the use of these flags for the mask is NOT officially documented
    CGEventMask eventMask =
        NSEventMaskGesture      |
        NSEventMaskMagnify      |
        NSEventMaskSwipe        |
        NSEventMaskRotate       |
        NSEventMaskBeginGesture |
        NSEventMaskEndGesture;

    // Create event tap
    eventTap = CGEventTapCreate(kCGHIDEventTap,           // Event tap is placed at the point where HID system events enter the window server
                                kCGHeadInsertEventTap,    // Event tap should be inserted before any pre-existing event tap at the same location
                                kCGEventTapOptionDefault, // Event tap is an active filter
                                eventMask,                // Event mask (as specified above)
                                cgEventCallbackFunction,  // Callback-Function
                                NULL);                    // User data (passed as last parameter to the Callback-Function)

    if(!eventTap)
    {
        [lbStatus setStringValue:@"Error"];
        NSLog(@"eventTap not valid");
        return;
    }

    // Create a run loop source
    runLoopSource = CFMachPortCreateRunLoopSource(kCFAllocatorDefault, eventTap, (CFIndex)0);

    // Add run loop source to current run loop
    CFRunLoopAddSource(CFRunLoopGetCurrent(), runLoopSource, kCFRunLoopCommonModes);

    // Enable event tap
    CGEventTapEnable(eventTap, true);

    [lbStatus setStringValue:@"Running"];
}


@end
