package de.test.widget;

import java.text.SimpleDateFormat;
import java.util.Date;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.SystemClock;
import android.widget.RemoteViews;

/**
 * Implements an AppWidget provider
 */
public class WidgetProvider extends AppWidgetProvider {
	public static final String URI_SCHEME = "test_widget";

	/**
	 * Called when the widget is updated. The current time (hh, mm, ss) is
	 * displayed.
	 */
	@Override
	public void onUpdate(Context context, AppWidgetManager appWidgetManager,
			int[] appWidgetIds) {
		// iterating over the widgets
		for (int appWidgetId : appWidgetIds) {

			RemoteViews remoteView = new RemoteViews(context.getPackageName(),
					R.layout.widget);

			Date now = new Date();
			SimpleDateFormat hour = new SimpleDateFormat("HH");
			SimpleDateFormat minute = new SimpleDateFormat("mm");
			SimpleDateFormat second = new SimpleDateFormat("ss");

			remoteView.setTextViewText(R.id.hours, hour.format(now).toString());
			remoteView.setTextViewText(R.id.minutes, minute.format(now)
					.toString());
			remoteView.setTextViewText(R.id.seconds, second.format(now)
					.toString());

			// updating the UI
			appWidgetManager.updateAppWidget(appWidgetId, remoteView);

		}
	}

	/**
	 * Called when the widget is deleted, the preferences are deleted and the
	 * alarm is cancelled.
	 */
	@Override
	public void onDeleted(Context context, int[] appWidgetIds) {
		// iterating over the widgets
		for (int appWidgetId : appWidgetIds) {

			// stop alarm
			Intent widgetUpdate = new Intent();
			widgetUpdate.setAction(AppWidgetManager.ACTION_APPWIDGET_UPDATE);
			widgetUpdate.putExtra(AppWidgetManager.EXTRA_APPWIDGET_ID,
					appWidgetId);
			widgetUpdate.setData(Uri.withAppendedPath(Uri.parse(URI_SCHEME
					+ "://widget/id/"), String.valueOf(appWidgetId)));
			PendingIntent newPending = PendingIntent.getBroadcast(context, 0,
					widgetUpdate, PendingIntent.FLAG_UPDATE_CURRENT);

			AlarmManager alarms = (AlarmManager) context
					.getSystemService(Context.ALARM_SERVICE);
			alarms.cancel(newPending);

			// remove preference
			SharedPreferences config = context.getSharedPreferences(
					WidgetConfiguration.PREFS_NAME, 0);
			SharedPreferences.Editor configEditor = config.edit();

			configEditor.remove(String.format(
					WidgetConfiguration.PREFS_UPDATE_RATE_FIELD_PATTERN,
					appWidgetId));
			configEditor.commit();
		}

		super.onDeleted(context, appWidgetIds);
	}

	/** 
	 * 
	 */
	@Override
	public void onReceive(Context context, Intent intent) {
		final String action = intent.getAction();

		// if the widget is to be updated...
		if (AppWidgetManager.ACTION_APPWIDGET_UPDATE.equals(action)) {

			// if the scheme doesn't match, that means it wasn't from the alarm
			// either it's the first time in (even before the configuration
			// is done) or after a reboot or update
			if (!URI_SCHEME.equals(intent.getScheme())) {
				final int[] appWidgetIds = intent.getExtras().getIntArray(
						AppWidgetManager.EXTRA_APPWIDGET_IDS);

				// iterating over the widgets
				for (int appWidgetId : appWidgetIds) {

					// retrieving the update rate
					SharedPreferences config = context.getSharedPreferences(
							WidgetConfiguration.PREFS_NAME, 0);
					int updateRateSeconds = config
							.getInt(
									String
											.format(
													WidgetConfiguration.PREFS_UPDATE_RATE_FIELD_PATTERN,
													appWidgetId), -1);

					if (updateRateSeconds != -1) {
						Intent widgetUpdate = new Intent();
						widgetUpdate
								.setAction(AppWidgetManager.ACTION_APPWIDGET_UPDATE);
						widgetUpdate.putExtra(
								AppWidgetManager.EXTRA_APPWIDGET_IDS,
								new int[] { appWidgetId });

						widgetUpdate.setData(Uri.withAppendedPath(Uri
								.parse(WidgetProvider.URI_SCHEME
										+ "://widget/id/"), String
								.valueOf(appWidgetId)));
						PendingIntent newPending = PendingIntent.getBroadcast(
								context, 0, widgetUpdate,
								PendingIntent.FLAG_UPDATE_CURRENT);

						AlarmManager alarms = (AlarmManager) context
								.getSystemService(Context.ALARM_SERVICE);
						alarms.setRepeating(AlarmManager.ELAPSED_REALTIME,
								SystemClock.elapsedRealtime(),
								updateRateSeconds * 1000, newPending);
					}
				}
			}
		}
		super.onReceive(context, intent);
	}
}
