package de.test.widget;

import android.app.Activity;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.appwidget.AppWidgetManager;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.Bundle;
import android.os.SystemClock;
import android.view.View;
import android.widget.EditText;
import android.widget.Toast;

/**
 * Configuration-activity that is displayed when the widget is added to a screen
 */
public class WidgetConfiguration extends Activity {
	public static final String PREFS_NAME = "TestWidgetPrefs";
	public static final String PREFS_UPDATE_RATE_FIELD_PATTERN = "UpdateRate-%d";
	private static final int PREFS_UPDATE_RATE_DEFAULT = 5;

	private int appWidgetId = AppWidgetManager.INVALID_APPWIDGET_ID;

	/**
	 * Called when the activity is created.
	 */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);

		// asking for the app widget id
		Intent launchIntent = getIntent();
		Bundle extras = launchIntent.getExtras();
		if (extras != null) {
			appWidgetId = extras.getInt(AppWidgetManager.EXTRA_APPWIDGET_ID,
					AppWidgetManager.INVALID_APPWIDGET_ID);

			Intent cancelResultValue = new Intent();
			cancelResultValue.putExtra(AppWidgetManager.EXTRA_APPWIDGET_ID,
					appWidgetId);
			setResult(RESULT_CANCELED, cancelResultValue);
		} else {
			finish();
		}

		setContentView(R.layout.widgetconfig);
		WidgetConfiguration.this.setTitle("Config");
	}

	/**
	 * OnClickListener for the save-button
	 */
	public void saveClick(View v) {
		final SharedPreferences config = getSharedPreferences(PREFS_NAME, 0);
		final EditText updateRateEntry = (EditText) findViewById(R.id.update_rate_entry);

		// checking if the string in the EditText element can be parsed as an integer
		int updateRateSeconds = PREFS_UPDATE_RATE_DEFAULT - 1;
		try {
			updateRateSeconds = Integer.parseInt(updateRateEntry.getText()
					.toString());
		} catch(NumberFormatException nFE) {
			Toast.makeText(WidgetConfiguration.this,
					"Update rate is not a string", Toast.LENGTH_SHORT)
					.show();
			return;
		}

		// checking if the update rate is ok
		if (updateRateSeconds < PREFS_UPDATE_RATE_DEFAULT) {
			Toast.makeText(WidgetConfiguration.this,
					"Update rate is too short (min 5 sec)", Toast.LENGTH_SHORT)
					.show();
			return;
		}

		// store off the user setting for update timing
		SharedPreferences.Editor configEditor = config.edit();

		configEditor.putInt(String.format(PREFS_UPDATE_RATE_FIELD_PATTERN,
				appWidgetId), updateRateSeconds);
		configEditor.commit();

		if (appWidgetId != AppWidgetManager.INVALID_APPWIDGET_ID) {

			// tell the app widget manager that it is now configured
			Intent resultValue = new Intent();
			resultValue.putExtra(AppWidgetManager.EXTRA_APPWIDGET_ID,
					appWidgetId);
			setResult(RESULT_OK, resultValue);

			// creating an intent which may start the widget itself
			Intent widgetUpdate = new Intent();
			widgetUpdate.setAction(AppWidgetManager.ACTION_APPWIDGET_UPDATE);
			widgetUpdate.putExtra(AppWidgetManager.EXTRA_APPWIDGET_IDS,
					new int[] { appWidgetId });

			// make this pending intent unique
			widgetUpdate.setData(Uri.withAppendedPath(Uri
					.parse(WidgetProvider.URI_SCHEME + "://widget/id/"), String
					.valueOf(appWidgetId)));
			PendingIntent newPending = PendingIntent.getBroadcast(
					getApplicationContext(), 0, widgetUpdate,
					PendingIntent.FLAG_UPDATE_CURRENT);

			// schedule the new widget for updating
			AlarmManager alarms = (AlarmManager) getApplicationContext()
					.getSystemService(Context.ALARM_SERVICE);
			alarms.setRepeating(AlarmManager.ELAPSED_REALTIME, SystemClock
					.elapsedRealtime(), updateRateSeconds * 1000, newPending);
		}
		// activity is now done
		finish();
	}
}
