//
//  TouchPadEventVisualizerAppDelegate.m
//  TouchPadEventVisualizer
//
//  Created by Gero Herkenrath on 4/13/11.
//  Copyright 2011 RWTH Aachen. All rights reserved.
//

#import "TouchPadEventVisualizerAppDelegate.h"
#import <QuartzCore/QuartzCore.h>

#pragma mark low-level C declarations
// these are low level functions doing the actual monitoring work
CGEventRef eventOccurred(CGEventTapProxy proxy, CGEventType type, CGEventRef event, void* refcon);
CFMachPortRef createEventTap(void *objectToBeInformed);


@interface TouchPadEventVisualizerAppDelegate (PrivateMethods)

- (void)setupTouchesView;
- (void)processTouchEventSet:(NSSet*)theTouches;

@end

#pragma mark low-level C implementation

CFMachPortRef createEventTap(void *objectToBeInformed) {  
	/* Each of the next two lines sets the event tap so it gets all events. We are only interested in a specific type (gestures, i.e. input
	 * from the touch pad). To set the tap for different event types, see CGEventTypes.h and below (how to get an event type mask from a type).
	 * Keep in mind that our example, checking for gestures, is still a bit whacky, see the next comments.
	 */
//	CGEventMask eventMask = NSAnyEventMask; // this works, but the next line is probably safer (see long comment below)
//	CGEventMask eventMask = kCGEventMaskForAllEvents; 	
	
	/* Note that this works on the CG level (the macro CGEventMaskBit is defined in CGEventTypes.h), even though NSEventTypeGesture is not defined in that header
	 * not is it included in some obvious way. It seems they're not defined in dependence of another, but still are defined the same way (as 29, to be exact).
	 * I highly doubt this will ever change in future releases of Mac OS X, because the usual event handling probably also relies on those two to be the same. 
	 * If they change it, they need to add a specialized conversion at some point that only works for these events. You still bypass this by catching all events,
	 * converting them into NSEvents and then checking their type. That seems kinda pointless in my eyes...
	 * Technically speaking, we're using private framework info here (since the CG equivalent of NSEventTypeGesture is probably defined in the implementation file
	 * and not the header), but at least we're not dynamically loading something.
	 */
	CGEventMask eventMask = CGEventMaskBit(NSEventTypeGesture);

	/* the next two lines work as well, alternatively, too, since the NS version of the masks are defined in the same way the macro constructs them on the CG level
	 */
//	CGEventMask eventMask = NSEventMaskGesture;
//	CGEventMask eventMask = NSEventMaskFromType(NSEventTypeGesture);

	
	/* doesn't matter for us here, I think this is somehow relevant for keyboard events, if you play around with this and it doesn't work, try
	 * using this code to check whether accessibility is on or off (and try turning it on/off in the System preferences under 
	 * "Universal Access - Enable access for assistive devices"). See Quartz Event Services Reference.
	 */
//	if (!AXAPIEnabled() && !AXIsProcessTrusted()) { 
//		printf("axapi not enabled");  
//	}
	
	return CGEventTapCreate(kCGSessionEventTap, 
							kCGHeadInsertEventTap, 
							kCGEventTapOptionDefault, 
							eventMask, 
							eventOccurred, 
							objectToBeInformed); 
}

/* Note: This callback allows monitoring events before they are passed to applications (depending on how the tap was set up).
 * It can also modify the events, but the way the calling function (somewhere in the window server) manages this is a bit non-obvious:
 * It does not really give you total control. If you modify the passed event and return it, your modifications are ignored unless the app
 * is running as root or "allow assistive devices..." is enabled in the System Preferences. In that case, you may also return NULL, then
 * the window server to deletes that event. HOWEVER, this seems not to work with gesture events, as far as I have tested it. It does process
 * the gesture event no matter what! I assume this is also the case for certain other events or key event combinations (to avoid blocking/
 * freezing the system's UI permanently), but I won't test that out. Still, you can create other events and return them. In this case they
 * get executed ADDITIONALLY to the original event. For example, you could post a keypress event everytime you do a gesture.
 */
CGEventRef eventOccurred(CGEventTapProxy proxy, CGEventType type, CGEventRef event, void* refcon) {
	if (type == NSEventTypeGesture) {
//		NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init]; // if called outside a regular Cocoa application, we might not have an autorelease pool here...
		
		/* example on how to post a keypress (here, the key z) everytime you do a gesture on the touchpad,
		 * note that this is lacking a matching key up event (i.e. not complete):
		 */
//		event = CGEventCreateKeyboardEvent (NULL, (CGKeyCode)6, true);
//		return event;
		
		// create an NSEvent to better handle the event
		NSEvent *myEvent = [NSEvent eventWithCGEvent: event];
		//		NSLog(@"%@", myEvent);
		NSSet *testSet = nil;
		// watch out for potentially weird events not caught by type check (apple doc is not detailed about this...)
		@try {
			testSet = [myEvent touchesMatchingPhase:NSTouchPhaseAny inView:nil];
		}
		@catch (NSException * e) {
			// NSLog(@"Exception occured, info: %@, reason: %@\n\n", [e name], [e reason]);
			testSet = nil;
		}
		@finally {
			// nothing to do here, yet...
		}
		
		if (testSet != nil) {
			// call object handler here to display touches
			[(id) refcon processTouchEventSet:testSet];
		}
		
//		[pool drain]; // see above...
	}
	
	// we just watch, so let's leave the event in peace...
	return event;
}

@implementation TouchPadEventVisualizerAppDelegate

@synthesize window, touchesView, catchEventsCheckBox;

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification {
	// Insert code here to initialize your application
	
	// we'll set up our event tap here and enable it if the check box says so
	tap = createEventTap(self);
	
	if (tap) {
		CFRunLoopSourceRef rl = CFMachPortCreateRunLoopSource(kCFAllocatorDefault, tap, 0);
		CFRunLoopAddSource(CFRunLoopGetMain(), rl, kCFRunLoopCommonModes);
		CFRelease(rl);
		if ([catchEventsCheckBox state] == NSOnState) // of course in other contexts, you may just always enable the tap
			CGEventTapEnable(tap, true);
//		CFRunLoopRun(); // only needed if you don't let your Cocoa app handle the run loop stuff
		
//		printf("Tap created.\n"); // just debug code
//		sleep(-1); // again, only needed in non-Cocoa stuff (e.g. your own simple main function)
	} else {
//		printf("failed!\n"); // won't happen ^^
	}
}

- (void)applicationWillTerminate:(NSNotification *)notification {
	/* this is technically not necessary since the tap will be released anyways when the app quits
	 * in other contexts, you may put this code into a dealloc (for example if you write your own
	 * wrapper class around it)
	 */
	CGEventTapEnable(tap, false); // just in case your callback needs to do some cleanup and reacts to tap disabled events
	CFRelease(tap);
}

- (void)awakeFromNib {
	[self setupTouchesView];	
}

- (IBAction)setEventGrabbing:(id)sender {
	if ([catchEventsCheckBox state] == NSOnState) {
		CGEventTapEnable(tap, true);
	} else {
		CGEventTapEnable(tap, false);
	}
}

#pragma mark private method implementation
- (void)setupTouchesView {
    CALayer *layer = [CALayer layer]; 
    CGColorSpaceRef colorSpace = CGColorSpaceCreateWithName(kCGColorSpaceGenericRGB);
    CGFloat components[4] = {0.0f, 0.0f, 0.0f, 1.0f};
    CGColorRef blackColor = CGColorCreate(colorSpace, components);
    layer.backgroundColor = blackColor; 
    [touchesView setLayer:layer]; 
    [touchesView setWantsLayer:YES];
    CGColorRelease(blackColor);
    CGColorSpaceRelease(colorSpace);
}

- (void)processTouchEventSet:(NSSet*)theTouches {
//	NSLog(@"object descr.: %@\n", theTouches); // debug code
	
	// note for code nazis: I know always creating and deleting layer objects just to draw some spots
	// is probably terribly inefficient/unelegant. I just did it cause it's easier and is more
	// flexible to play with the events. You could control animations from here, too, I just didn't
	// get any nice ideas other than red dots.
	CALayer *layer = [touchesView layer];
	if (layer == nil)
		return;
	NSArray *oldlayers = [layer sublayers];
	int i, count = [oldlayers count];
	for (i = 0; i < count; ++i) { // note: don't use fast enumeration for removing those... -_-
		[[oldlayers objectAtIndex:i] removeFromSuperlayer];
	}

	
	CGRect frame = layer.bounds;
	CGColorSpaceRef colorSpace = CGColorSpaceCreateWithName(kCGColorSpaceGenericRGB);
	CGFloat components[4] = {1.0f, 0.0f, 0.0f, 1.0f};
	CGColorRef redColor = CGColorCreate(colorSpace, components);
	
	for (NSTouch *myTouch in theTouches) {
		// note: sometimes you get touches with an unknown phase, apparently even after the last finger has been
		// moved from the touchpad. that results in a dot being drawn even with the hand not on the pad.
		// to prevent that, I manually check for all the touch phase types that definitely have a hand on the pad
		// and only draw those. Sometimes that still happens, i.e. sometimes an event of one of the types below
		// gets cought by the event tap when the hand already left the device. That type is "unkown" at this
		// level of the api, I assume it's something like those "proximity" events used for tablets. 
		if (([myTouch phase] != NSTouchPhaseBegan) && ([myTouch phase] != NSTouchPhaseMoved)
			&& ([myTouch phase] != NSTouchPhaseStationary) && ([myTouch phase] != NSTouchPhaseTouching)) {
			continue;
		}
		CGRect subframe = frame;
		subframe.origin.x = frame.size.width * [myTouch normalizedPosition].x;
		subframe.origin.y = frame.size.height * [myTouch normalizedPosition].y;
		subframe.size.height /= 20.0f;
		subframe.size.width = subframe.size.height;
		CALayer *sublayer = [CALayer layer];
		sublayer.frame = subframe;
		sublayer.cornerRadius = subframe.size.width / 2.0f;
		sublayer.backgroundColor = redColor;
		[layer addSublayer:sublayer];
	}
	CGColorRelease(redColor);
	CGColorSpaceRelease(colorSpace);
}

@end
