//
//  TouchView.m
//  TouchViz
//
//  Created by Jonathan Diehl on 25.11.10.
//  Copyright 2010 RWTH. All rights reserved.
//

#define BUFFER_SIZE 20


#import "TouchView.h"
#import "NSObject+ObjectKey.h"
#import "RingBuffer.h"


@interface TouchView ()
- (void)setup;
- (void)addTouches:(NSSet *)touches began:(BOOL)began;
- (void)removeTouches:(NSSet *)touches;
@end


@implementation TouchView


#pragma mark UIView event handling

// touch began -> create path
- (void)touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event;
{
	[self addTouches:touches began:YES];
}

// touch moved -> add points to path
- (void)touchesMoved:(NSSet *)touches withEvent:(UIEvent *)event;
{
	[self addTouches:touches began:NO];
}

// touch canceled -> remove path
- (void)touchesCancelled:(NSSet *)touches withEvent:(UIEvent *)event;
{
	[self removeTouches:touches];
}

// touch ended -> remove path
- (void)touchesEnded:(NSSet *)touches withEvent:(UIEvent *)event;
{
	[self removeTouches:touches];
}


#pragma mark UIView drawing

// draw
- (void)drawRect:(CGRect)rect
{
	size_t point_size = sizeof(CGPoint);
	
	NSString *key;
	RingBuffer *path;
	CGPoint *p1, *p2;
	CGFloat col;
	CGFloat col_step = 1.0 / BUFFER_SIZE;
	
	// get context
    CGContextRef context = UIGraphicsGetCurrentContext();
	
	// configure line
	CGContextSetLineWidth(context, 3.0);
	CGContextSetLineJoin(context, kCGLineJoinRound);
	
	// draw touch paths
	for(key in paths) {
		path = [paths objectForKey:key];
		
		// skip short paths
		if(path.length < 2) continue;
		
		// copy the buffer from the path
		[path copyBuffer:buffer];
		
		// get the first point
		p1 = buffer;
		col = (BUFFER_SIZE - path.length) * col_step;
		
		// go through all points and draw lines
		for(int i = 1; i < path.length; i++) {
			p2 = p1;
			p1 = buffer + i*point_size;
			
			// construct line
			CGContextMoveToPoint(context, p1->x, p1->y);
			CGContextAddLineToPoint(context, p2->x, p2->y);
			
			// draw
			col += col_step;
			CGContextSetRGBStrokeColor(context, col, col, col, 1.0);
			CGContextStrokePath(context);
		}
		
	}
}


#pragma mark init & cleanup

// load from nib file
- (void)awakeFromNib;
{
	[self setup];
}

// designated init
- (id)initWithFrame:(CGRect)frame
{
	self = [super initWithFrame:frame];
	if (self) {
		[self setup];
	}
	return self;
}


// cleanup
- (void)dealloc
{
	[paths release];
	free(buffer);
	[super dealloc];
}


#pragma mark private methods

- (void)setup;
{
	// enable multi touch
	self.multipleTouchEnabled = YES;
	
	// set up buffers
	paths = [[NSMutableDictionary alloc] initWithCapacity:5];
	buffer = malloc(BUFFER_SIZE * sizeof(CGPoint));
}

- (void)addTouches:(NSSet *)touches began:(BOOL)began;
{
	UITouch *touch;
	RingBuffer *path;
	CGPoint point;
	
	// go through all touches
	for(touch in touches) {
		
		// if first touch: create path and add to our dictionary
		if(began) {
			path = [[RingBuffer alloc] initWithSize:sizeof(CGPoint) capacity:BUFFER_SIZE];
			[paths setObject:path forKey:[touch objectKey]];
			[path release];
		}
		
		// otherwise: get existing path
		else {
			path = [paths objectForKey:[touch objectKey]];
		}
		
		// add point
		point = [touch locationInView:self];
		[path push:&point];
	}
	
	// redraw
	[self setNeedsDisplay];
}

- (void)removeTouches:(NSSet *)touches;
{
	UITouch *touch;
	
	// remove paths for touches
	for(touch in touches) {
		[paths removeObjectForKey:[touch objectKey]];
	}
	
	// redraw
	[self setNeedsDisplay];
}



@end
