//
//  MultiFingerGestureController.m
//  GestureRecognizers
//
//  Created by Jan-Peter Krämer on 31.08.10.
//  Copyright 2010 RWTH Aachen University. All rights reserved.
//

#import "MultiFingerGestureController.h"
#import "UpDownGestureRecognizer.h" 

// private methods
@interface MultiFingerGestureController ()

-(void)singleTapDetected:(UIGestureRecognizer*)sender;
-(void)doubleTapDetected:(UIGestureRecognizer*)sender;
-(void)swipeDetected:(UIGestureRecognizer*)sender;

@end


@implementation MultiFingerGestureController

@synthesize view, delegate;

// designated init
-(id)initWithView:(UIView *)aView;
{
	if (self = [super init]) {
		self.view = aView;
		gestureRecognizers = [NSMutableArray arrayWithCapacity:31]; 
		
	}
	return self;
}

// cleanup
- (void)dealloc
{
	self.view = nil;
	self.delegate = nil;
	[gestureRecognizers release];
	[super dealloc];
}


#pragma mark accessors

// update the view -> attach gesture recognizers
- (void)setView:(UIView *)aView; 
{
	if(aView == self.view) return; 
	
	// remove previous gesture recognizers
	if([gestureRecognizers count] > 0) {
		UIGestureRecognizer *recognizer;
		for(recognizer in gestureRecognizers) {
			[self.view removeGestureRecognizer:recognizer];
		}
		[gestureRecognizers removeAllObjects];
	}
	
	// update view
	[view autorelease];
	view = [aView retain];
	
	// we are done if the new view is nil
	if(!view) return;
	
	// all gesture recognizers are realized for 1-5 simultaneous touches
	for (int i = 1; i<=5; i++){
		
		// single tap
		UITapGestureRecognizer *tapRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(singleTapDetected:)]; 
		tapRecognizer.numberOfTouchesRequired = i; 
		tapRecognizer.cancelsTouchesInView = NO; 
		tapRecognizer.delaysTouchesBegan = NO; 
		tapRecognizer.delaysTouchesEnded = NO; 	
		[view addGestureRecognizer:tapRecognizer]; 
		[gestureRecognizers addObject:tapRecognizer]; 
		[tapRecognizer release]; 
		
		// double tap
		UITapGestureRecognizer *doubleTapRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(doubleTapDetected:)]; 
		doubleTapRecognizer.numberOfTouchesRequired = i; 
		doubleTapRecognizer.numberOfTapsRequired = 2;
		doubleTapRecognizer.cancelsTouchesInView = NO; 
		doubleTapRecognizer.delaysTouchesBegan = NO; 
		doubleTapRecognizer.delaysTouchesEnded = NO; 
		[view addGestureRecognizer:doubleTapRecognizer]; 
		[gestureRecognizers addObject:doubleTapRecognizer]; 
		[doubleTapRecognizer release]; 
		
		// swipe in every direction
		for (int j = 1; j < (1<<4); j = j << 1){
			UISwipeGestureRecognizer *swipeRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(swipeDetected:)]; 
			swipeRecognizer.numberOfTouchesRequired = i; 
			swipeRecognizer.direction = j; 
			swipeRecognizer.cancelsTouchesInView = NO; 
			swipeRecognizer.delaysTouchesBegan = NO; 
			swipeRecognizer.delaysTouchesEnded = NO; 		
			[view addGestureRecognizer:swipeRecognizer]; 
			[gestureRecognizers addObject:swipeRecognizer]; 
			[swipeRecognizer release]; 		
		}
	}
	
	// custom up-down swipe recognizer
	UpDownGestureRecognizer *upDownRecognizer = [[UpDownGestureRecognizer alloc] initWithTarget:self action:@selector(upDownDetected:)]; 
	upDownRecognizer.numberOfTouchesRequired = 1; 
	upDownRecognizer.cancelsTouchesInView = NO; 
	upDownRecognizer.delaysTouchesBegan = NO; 
	upDownRecognizer.delaysTouchesEnded = NO; 
	[view addGestureRecognizer:upDownRecognizer]; 
	[gestureRecognizers addObject:upDownRecognizer]; 
	[upDownRecognizer release]; 
}


#pragma mark gesture detection notifications

// UITapGestureRecognizer
- (void)singleTapDetected:(UIGestureRecognizer *)sender; 
{
	[self.delegate recognizedGestureType:GestureTypeTap withNumberOfTouches:[sender numberOfTouches] inDirection:GestureDirectionNone]; 
}

// UITapGestureRecognizer
- (void)doubleTapDetected:(UIGestureRecognizer *)sender; 
{
	[self.delegate recognizedGestureType:GestureTypeDoubleTap withNumberOfTouches:[sender numberOfTouches] inDirection:GestureDirectionNone]; 
}

// UISwipeGestureRecognizer
- (void)swipeDetected:(UIGestureRecognizer *)sender; 
{
	GestureDirection resultDirection; 
	switch (((UISwipeGestureRecognizer*)sender).direction) {
			
		// up
		case UISwipeGestureRecognizerDirectionUp:
			resultDirection = GestureDirectionUp; 
			break;
			
		// right
		case UISwipeGestureRecognizerDirectionRight:
			resultDirection = GestureDirectionRight;
			break;
			
		// left
		case UISwipeGestureRecognizerDirectionLeft:
			resultDirection = GestureDirectionLeft;
			break;
			
		// down
		case UISwipeGestureRecognizerDirectionDown:
			resultDirection = GestureDirectionDown;
			break;

		default:
			return;
	}
	
	[self.delegate recognizedGestureType:GestureTypeSwipe withNumberOfTouches:[sender numberOfTouches] inDirection:resultDirection]; 
}

// UIUpDownGestureRecognizer
- (void)upDownDetected:(UIGestureRecognizer *)recognizer;
{
	GestureType type;
	switch(recognizer.state) {
			
		// touch down
		case UIGestureRecognizerStateBegan:
			type = GestureTypeTouchDown;
			break;
			
		// touch moved
		case UIGestureRecognizerStateChanged:
			// nothing
			return;
			
		// touch up
		case UIGestureRecognizerStateEnded:
		case UIGestureRecognizerStateCancelled:
			type = GestureTypeTouchUp;
			break;
			
		default:
			return;
	}
	
	[self.delegate recognizedGestureType:type withNumberOfTouches:[recognizer numberOfTouches] inDirection:GestureDirectionNone]; 
}


@end
