//
//  Path.m
//  TouchViz
//
//  Created by Jonathan Diehl on 25.11.10.
//  Copyright 2010 RWTH. All rights reserved.
//

#import "RingBuffer.h"


@interface RingBuffer ()
- (void *)step:(void *)pointer;
@end


@implementation RingBuffer

@synthesize length;


#pragma mark data manipulation

// add item
- (void)push:(void *)item;
{
	// pop if buffer is full
	if(length >= capacity) {
		[self pop];
	}
	
	// copy item to buffer
	memcpy(head, item, size);

	// step the head
	head = [self step:head];
	length++;
}

// retrieve item
- (void *)pop;
{
	// buffer must not be empty
	NSAssert(length > 0, @"tried to pop an empty buffer");
	
	void *item = tail;
	
	// step the tail
	tail = [self step:tail];
	length--;
	
	return item;
}


#pragma mark conversion

- (void)copyBuffer:(void *)copy;
{
	// tail is before head -> simple copy
	if(tail < head) {
		memcpy(copy, tail, length*size);
	}
	
	// head is before tail -> copy in two chunks
	else {
		size_t chunk_size = buffer_end-tail;
		memcpy(copy, tail, chunk_size);
		memcpy(copy+chunk_size, buffer, head-buffer);
	}
}


#pragma mark init & cleanup

// init
- (id)initWithSize:(size_t)aSize capacity:(NSUInteger)aCapacity;
{
	self = [super init];
	if(self != nil) {
		capacity = aCapacity;
		size = aSize;
		buffer = malloc(capacity*size);
		buffer_end = buffer + capacity*size;
		length = 0;
		head = buffer;
		tail = buffer;
	}
	return self;
}

// cleanup
- (void) dealloc
{
	free(buffer);
	[super dealloc];
}


#pragma mark debug description

- (NSString *)description;
{
	return [NSString stringWithFormat:@"<RingBuffer head:%d tail:%d length:%d>", (head-buffer)/size, (tail-buffer)/size, length];
}


#pragma mark private methods

// step the pointer forward
- (void *)step:(void *)pointer;
{
	pointer += size;
	if(pointer >= buffer_end) pointer = buffer;
	return pointer;
}


@end
