//
//  GestureRecognizersViewController.m
//  GestureRecognizers
//
//  Created by Jan-Peter Krämer on 31.08.10.
//  Copyright 2010 RWTH Aachen University. All rights reserved.
//

#import "GestureRecognizersViewController.h"
#import "MultiFingerGestureController.h"


#pragma mark helper methods

NSString *gestureDirectionToString(GestureDirection direction) {
	switch (direction) {
		case GestureDirectionUp:
			return @"Up"; 
		case GestureDirectionDown:
			return @"Down"; 
		case GestureDirectionLeft:
			return @"Left"; 
		case GestureDirectionRight:
			return @"Right"; 
		case GestureDirectionNone:
			return @"None"; 
	}
	return @"unknown";
}

NSString *gestureTypeToString(GestureType type) {
	switch (type) {
		case GestureTypeTap:
			return @"Tap";
		case GestureTypeDoubleTap:
			return @"DoubleTap";
		case GestureTypeSwipe:
			return @"Swipe";
		case GestureTypeTouchDown:
			return @"TouchDown";
		case GestureTypeTouchUp:
			return @"TouchUp";
	}
	return @"unknown";
}

@implementation GestureRecognizersViewController

@synthesize gestureController, gestureTextLayer; 


#pragma mark UIViewController

// init
- (void)viewDidLoad
{
    [super viewDidLoad];
	
	// enable multi touch
	self.view.multipleTouchEnabled = YES;
	
	// set up gesture congtroller
	self.gestureController = [[[MultiFingerGestureController alloc] initWithView:self.view] autorelease];
	self.gestureController.delegate = self;
	
	// set up gesture text layer
	self.gestureTextLayer = [CATextLayer layer]; 
	self.gestureTextLayer.frame = self.view.frame;	
	self.gestureTextLayer.fontSize = 48.0;
	self.gestureTextLayer.alignmentMode = kCAAlignmentCenter;
	self.gestureTextLayer.foregroundColor = [[UIColor blackColor] CGColor];
	[self.view.layer addSublayer:self.gestureTextLayer];
}

// cleanup
- (void)dealloc
{
	self.gestureTextLayer = nil; 
	self.gestureController = nil;
    [super dealloc];
}

// allow rotation
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation; 
{
	return YES;
}

// will rotate
- (void)willRotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
	CGRect frame = self.view.frame;
	if(UIInterfaceOrientationIsLandscape(toInterfaceOrientation)) {
		frame.size = CGSizeMake(frame.size.height, frame.size.width);
	} else {
	}
	
	// gesture text layer
	self.gestureTextLayer.frame = frame;
}


#pragma mark MultiFingerGestureControllerDelegate

// recognized a gesture
-(void)recognizedGestureType:(GestureType)type withNumberOfTouches:(int)fingerCount inDirection:(GestureDirection)direction; 
{
	NSString *typeString = gestureTypeToString(type);
	NSString *directionString = gestureDirectionToString(direction);
	NSString *gestureString = [NSString stringWithFormat:@"\n%@ %d %@", typeString, fingerCount, directionString];

	// logging
	NSLog(@"Recognized Gesture: %@", gestureString);

	[CATransaction begin];
	[CATransaction disableActions]; 
	[self.gestureTextLayer removeAllAnimations];
	
	// update layer text
	if ([self.gestureTextLayer.string length] > 0) {
		self.gestureTextLayer.string = [self.gestureTextLayer.string stringByAppendingString:gestureString];
	} else {
		self.gestureTextLayer.string = gestureString;
	}
	
	// set opacity
	self.gestureTextLayer.opacity = 1.0;
	[CATransaction commit]; 
	
	// create fade-out animation
	CABasicAnimation *anim = [CABasicAnimation animationWithKeyPath:@"opacity"]; 
	anim.toValue = [NSNumber numberWithFloat:0.0]; 
	anim.fromValue = [NSNumber numberWithFloat:1.0]; 
	anim.duration = 1.0; 
	anim.delegate = self;
	anim.fillMode = kCAFillModeForwards; 
	anim.removedOnCompletion = NO;
	[self.gestureTextLayer addAnimation:anim forKey:@"opacityAnimation"]; 	
}


#pragma mark CAAnimation delegate methods

- (void)animationDidStop:(CAAnimation *)anim finished:(BOOL)finished; 
{
	if(!finished) return;
	
	// remove text from the gesture layer
	[CATransaction begin];
	[CATransaction disableActions]; 
	self.gestureTextLayer.string = @""; 
	[CATransaction commit];
}

@end
