//
//  KNKinect.m
//  KinectTest
//
//  Created by Chatchavan Wacharamanotham on 13/11/2010.
//  Copyright 2010 Media Computing Group, RWTH Aachen. All rights reserved.
//

#import "KNKinect.h"
#include "libfreenect.h"

#define KNDeptImageWidth 640
#define KNDeptImageHeight 480

KNKinect *selfReference;
id <KNKinectDelegate> delegateReference;

// =====================================================================================================================
@interface KNKinect()
// =====================================================================================================================

-(void)stopRefreshTimer;

void processRgbImage(uint8_t *buf, int width, int height);
void processDepthImage(uint16_t *buf, int width, int height);

@end
// =====================================================================================================================



// =====================================================================================================================
@implementation KNKinect
// =====================================================================================================================

@synthesize delegate;
@synthesize refreshInterval;

// ---------------------------------------------------------------------------------------------------------------------
#pragma mark -
#pragma mark init and dealloc
// ---------------------------------------------------------------------------------------------------------------------

-(id)initWithAnyDevice;
{
	self = [super init];
	if (self != nil) 
	{
		libusb_init(NULL);
		device = libusb_open_device_with_vid_pid(NULL, 0x45e, 0x2ae);
		
		if (!device)
		{
			[super dealloc];
			return nil;
		}
		
		// store reference for c function call
		selfReference = self;
		
		// default refresh interval
		refreshInterval = 0.01;
	}
	return self;
}

-(id)init;
{
	return nil;
}


-(void)dealloc;
{	
	[self stopRefreshTimer];
	[super dealloc];
}

// ---------------------------------------------------------------------------------------------------------------------
#pragma mark -
#pragma mark feed control
// ---------------------------------------------------------------------------------------------------------------------

-(void)start;
{
	int isSuccess;
	
	// claim device interface
	isSuccess = libusb_claim_interface(device, 0);
	if (isSuccess != 0)
	{
		NSLog(@"Cannot claim Kinect device interface.");
		return;
	}
	
	// start camera
	isSuccess = cams_init(device, processDepthImage, processRgbImage);
	if (isSuccess != 0)
	{
		// TODO: interprete return code of cams_init
	}
	
	// start timer (30 frame per second)
	refreshTimer = [[NSTimer timerWithTimeInterval:self.refreshInterval target:self selector:@selector(refreshTimerCallBack:) userInfo:nil repeats:YES] retain];
	[[NSRunLoop currentRunLoop] addTimer:refreshTimer forMode:NSRunLoopCommonModes];
	
	// tell delegate
	if ([delegate respondsToSelector:@selector(kinectDidStart:)])
	{
		[delegate kinectDidStart:self];
	}
	
}

-(void)stop;
{
	int isSuccess;
	
	// stop timer
	[self stopRefreshTimer];
	
	// TODO: stop camera
	
	// release device interface
	isSuccess = libusb_release_interface(device, 0);
	if (isSuccess !=0)
	{
		NSLog(@"Cannot release Kinect device interface.");
	}
	
	// tell delegate
	if ([delegate respondsToSelector:@selector(kinectDidStop:)])
	{
		[delegate kinectDidStop:self];
	}
	
}

// ---------------------------------------------------------------------------------------------------------------------
#pragma mark -
#pragma mark delegate
// ---------------------------------------------------------------------------------------------------------------------

-(void)setDelegate:(id <KNKinectDelegate>)value;
{
    if (delegate != value) 
	{
        delegate = value;
		delegateReference = delegate;
    }
}


// ---------------------------------------------------------------------------------------------------------------------
#pragma mark -
#pragma mark timer
// ---------------------------------------------------------------------------------------------------------------------

-(void)refreshTimerCallBack:(NSTimer *)theTimer;
{
	if (libusb_handle_events(NULL) != 0)
	{
		// TODO: handle event error
	}
}

-(void)stopRefreshTimer;
{
	[refreshTimer invalidate];
	[refreshTimer release];
	refreshTimer = nil;	
}

// ---------------------------------------------------------------------------------------------------------------------
#pragma mark -
#pragma mark libfreenect call-back
// ---------------------------------------------------------------------------------------------------------------------

void processDepthImage(uint16_t *buf, int width, int height)
{	
	// create depth image
	CGColorSpaceRef grayColorSpace = CGColorSpaceCreateDeviceGray();
	CGContextRef bitmapContext = CGBitmapContextCreate(buf, 
													   width, 
													   height, 
													   16, 
													   2 * width, 
													   grayColorSpace,
													   kCGImageAlphaNone);
	CGImageRef depthImage = CGBitmapContextCreateImage(bitmapContext);
	
	// tell delegate
	[delegateReference didUpdateDepth:depthImage kinect:selfReference];
	
	// cleanup
	CGImageRelease(depthImage);
	CFRelease(bitmapContext);
	CFRelease(grayColorSpace);
}

void processRgbImage(uint8_t *buf, int width, int height)
{
	// convert array to RGBA
	char* rgba = (char *)malloc(width * height * 4);
	for(int i = 0; i < width * height; i++)
	{
		rgba[4*i]	= buf[3*i];
		rgba[4*i+1] = buf[3*i+1];
		rgba[4*i+2] = buf[3*i+2];
		rgba[4*i+3] = 0;
	}
	
	// create rgb image
	CGColorSpaceRef rgbColorSpace = CGColorSpaceCreateDeviceRGB();
	CGContextRef bitmapContext = CGBitmapContextCreate(rgba, 
													   width, 
													   height, 
													   8, 
													   4 * width, 
													   rgbColorSpace,
													   kCGImageAlphaNoneSkipLast);
	CGImageRef rgbImage = CGBitmapContextCreateImage(bitmapContext);
	
	// tell delegate
	[delegateReference didUpdateRgb:rgbImage kinect:selfReference];
	
	// cleanup
	CGImageRelease(rgbImage);
	CFRelease(bitmapContext);
	CFRelease(rgbColorSpace);
	free(rgba);
}


@end
